package ru.yandex.direct.oneshot.oneshots.smartchangestrategy;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.oneshots.smartchangestrategy.repository.OneshotCampaignRepository;
import ru.yandex.direct.oneshot.oneshots.smartchangestrategy.validation.CampaignStrategyValidator;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER;

/**
 * Меняет стратегии AUTOBUDGET_AVG_CPA_PER_* на AUTOBUDGET_AVG_CPС_PER_* у смарт кампаний, у которых
 * AvgCpa/FilterAvgCpa=(null||0) и avgBid/FilterAvgBid>0
 * <p>
 * Выполняется перед {@link SmartCleanAvgBidInCpaStrategyIterativeOneshot}
 */
@Component
@Multilaunch
@Approvers({"ppalex"})
@ParametersAreNonnullByDefault
public class SmartChangeCpaStrategyToCpcOneshot extends SmartChangeStrategyBase {
    private static final Logger logger = LoggerFactory.getLogger(SmartChangeCpaStrategyToCpcOneshot.class);

    private final CampaignRepository campaignRepository;
    private final DslContextProvider dslContextProvider;
    private final CampaignStrategyValidator campaignStrategyValidator;

    @Autowired
    public SmartChangeCpaStrategyToCpcOneshot(CampaignRepository campaignRepository,
                                              OneshotCampaignRepository oneshotCampaignRepository,
                                              DslContextProvider dslContextProvider,
                                              BsResyncService bsResyncService) {
        super(oneshotCampaignRepository, dslContextProvider, campaignRepository, bsResyncService, logger);
        this.campaignRepository = campaignRepository;
        this.dslContextProvider = dslContextProvider;
        campaignStrategyValidator = new CampaignStrategyValidator(logger);
    }

    @Override
    List<Campaign> getCampaignsForMigration(int shard, Collection<Long> campaignIds) {

        Collection<Campaign> campaigns = campaignRepository.getCampaigns(dslContextProvider.ppc(shard),
                getCampaignsSelectionCriteria(campaignIds), false);

        // Отбираем только стратегии AUTOBUDGET_AVG_CPA_PER_* c FilterAvgBid/AvgBid и без FilterAvgCpa/AvgCpa
        return campaigns.stream()
                .filter(campaign -> {
                    StrategyData strategyData = campaign.getStrategy().getStrategyData();
                    if (campaign.getStrategy().getStrategyName().equals(AUTOBUDGET_AVG_CPA_PER_FILTER)) {
                        return strategyData.getFilterAvgBid() != null &&
                                strategyData.getFilterAvgBid().compareTo(BigDecimal.ZERO) > 0 &&
                                (strategyData.getFilterAvgCpa() == null || strategyData.getFilterAvgCpa().equals(BigDecimal.ZERO));
                    } else if (campaign.getStrategy().getStrategyName().equals(AUTOBUDGET_AVG_CPA_PER_CAMP)) {
                        return strategyData.getAvgBid() != null &&
                                strategyData.getAvgBid().compareTo(BigDecimal.ZERO) > 0 &&
                                (strategyData.getAvgCpa() == null || strategyData.getAvgCpa().equals(BigDecimal.ZERO));
                    }
                    return false;
                })
                .collect(Collectors.toList());
    }

    @Override
    boolean modifyAndValidateStrategy(int shard, Campaign campaign, DbStrategy strategyWas) {
        DbStrategy strategyNew = campaign.getStrategy();
        if (strategyNew.getStrategyName().equals(AUTOBUDGET_AVG_CPA_PER_FILTER)) {
            strategyNew.setStrategy(CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_FILTER);
            strategyNew.setStrategyName(AUTOBUDGET_AVG_CPC_PER_FILTER);
            strategyNew.getStrategyData()
                    .withName(StrategyName.toSource(AUTOBUDGET_AVG_CPC_PER_FILTER).name())
                    .withGoalId(null)
                    .withFilterAvgCpa(null);
        } else if (strategyNew.getStrategyName().equals(AUTOBUDGET_AVG_CPA_PER_CAMP)) {
            strategyNew.setStrategy(CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_CAMP);
            strategyNew.setStrategyName(AUTOBUDGET_AVG_CPC_PER_CAMP);
            strategyNew.getStrategyData()
                    .withName(StrategyName.toSource(AUTOBUDGET_AVG_CPC_PER_CAMP).name())
                    .withGoalId(null)
                    .withAvgCpa(null)
                    .withPayForConversion(null);
        }
        if (!campaignStrategyValidator.validateCampaignStrategy(shard, campaign)) {
            return false;
        }

        logger.info("shard: {}, cid: {}, update CAMPAIGNS.STRATEGY_NAME old: {} new: {}",
                shard, campaign.getId(), strategyWas.getStrategyName(),
                strategyNew.getStrategyName().name());
        logger.info("shard: {}, cid: {}, update CAMPAIGNS.STRATEGY_DATA old: {} new: {}",
                shard, campaign.getId(), JsonUtils.toJson(strategyWas.getStrategyData()),
                JsonUtils.toJson(strategyNew.getStrategyData()));
        logger.info("shard: {}, cid: {}, update CAMP_OPTIONS.STRATEGY old: {} new: {}",
                shard, campaign.getId(), strategyWas.getStrategy().name(),
                strategyNew.getStrategy().name());
        return true;
    }
}
