package ru.yandex.direct.oneshot.oneshots.smartchangestrategy;

import java.text.DecimalFormat;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.core.entity.campaign.container.CampaignsSelectionCriteria;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.base.ShardedIterativeOneshotWithoutInput;
import ru.yandex.direct.oneshot.oneshots.smartchangestrategy.repository.OneshotCampaignRepository;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority.PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
abstract class SmartChangeStrategyBase extends ShardedIterativeOneshotWithoutInput<SmartChangeStrategyState> {
    private static final int CHUNK_SIZE_FOR_UPDATE = 25;
    private static final int CHUNK_SIZE_CAMPAIGN_IDS = 20_000;

    private final DslContextProvider dslContextProvider;
    private final OneshotCampaignRepository oneshotCampaignRepository;
    private final CampaignRepository campaignRepository;
    private final BsResyncService bsResyncService;
    private final Logger logger;

    SmartChangeStrategyBase(OneshotCampaignRepository oneshotCampaignRepository,
                            DslContextProvider dslContextProvider,
                            CampaignRepository campaignRepository,
                            BsResyncService bsResyncService,
                            Logger logger) {
        this.oneshotCampaignRepository = oneshotCampaignRepository;
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
        this.bsResyncService = bsResyncService;
        this.logger = logger;
    }

    @Override
    protected SmartChangeStrategyState execute(@Nullable SmartChangeStrategyState state, int shard) {
        if (state == null) {
            var minAndMaxCampaignIds = oneshotCampaignRepository.getMinAndMaxSmartCampaignId(dslContextProvider, shard);
            if (minAndMaxCampaignIds.getLeft() == null) {
                logger.info("shard: {}, campaign table is empty", shard);
                return null;
            }
            state = new SmartChangeStrategyState()
                    .withFromCampaignId(minAndMaxCampaignIds.getLeft())
                    .withMaxCampaignId(minAndMaxCampaignIds.getRight());
        }
        long campaignIdFrom = state.getFromCampaignId();
        long campaignIdToExclude = state.getFromCampaignId() + CHUNK_SIZE_CAMPAIGN_IDS;
        DecimalFormat formatter = new DecimalFormat("#,###");
        logger.info("shard: {}, search for campaigns ranging from {} to {}, ending with {}", shard,
                formatter.format(campaignIdFrom), formatter.format(campaignIdToExclude - 1L),
                formatter.format(state.getMaxCampaignId()));

        List<Long> campaignIds = LongStream
                .range(campaignIdFrom, Math.min(campaignIdToExclude, state.getMaxCampaignId() + 1L))
                .boxed()
                .collect(Collectors.toList());

        List<Campaign> campaignsForMigration = getCampaignsForMigration(shard, campaignIds);

        if (!campaignsForMigration.isEmpty()) {
            logger.info("shard: {}, going to update strategies for {} campaigns", shard, campaignsForMigration.size());
            for (List<Campaign> chunk : Lists.partition(campaignsForMigration, CHUNK_SIZE_FOR_UPDATE)) {
                state.addAndGetCountOfUpdatedCampaigns(updateCampaignStrategies(shard, chunk));
            }
        }
        return getNextState(shard, state);
    }

    private SmartChangeStrategyState getNextState(int shard, SmartChangeStrategyState state) {
        if (state.getFromCampaignId() + CHUNK_SIZE_CAMPAIGN_IDS > state.getMaxCampaignId()) {
            logger.info("shard: {}, it was a last iteration, {} campaigns have been changed",
                    shard, state.getCountOfUpdatedCampaigns());
            return null;
        }
        return state.withFromCampaignId(state.getFromCampaignId() + CHUNK_SIZE_CAMPAIGN_IDS);
    }

    private int updateCampaignStrategies(int shard, List<Campaign> campaigns) {
        Set<Long> campaignIdsToResync = new HashSet<>();

        final AtomicInteger updated = new AtomicInteger();
        dslContextProvider.ppc(shard).transaction(configuration -> {
            DSLContext dslContext = configuration.dsl();

            List<Long> chunkCampaignIds = mapList(campaigns, Campaign::getId);
            List<Campaign> currentCampaigns = campaignRepository.getCampaigns(dslContext,
                    getCampaignsSelectionCriteria(chunkCampaignIds), true);
            Map<Long, Campaign> cidToCurrentCampaign = StreamEx.of(currentCampaigns)
                    .mapToEntry(Campaign::getId)
                    .invert()
                    .toMap();

            List<Campaign> campaignsToUpdate = campaigns.stream()
                    .filter(campaign -> {
                        if (!cidToCurrentCampaign.containsKey(campaign.getId()) ||
                                !Objects.equals(cidToCurrentCampaign.get(campaign.getId()).getStrategy(),
                                        campaign.getStrategy())) {
                            logger.warn("shard: {}, cid: {}, campaign has been changed, skipping it",
                                    shard, campaign.getId());
                            return false;
                        }

                        DbStrategy strategyWas = cidToCurrentCampaign.get(campaign.getId()).getStrategy();
                        return modifyAndValidateStrategy(shard, campaign, strategyWas);
                    })
                    .collect(Collectors.toList());

            updated.addAndGet((int) campaignsToUpdate.stream()
                    .filter(campaign -> oneshotCampaignRepository.updateCampaignStrategy(dslContext, campaign))
                    .count());
            campaignIdsToResync.addAll(mapList(campaignsToUpdate, Campaign::getId));
        });
        logger.info("shard: {}, updated {} campaigns", shard, updated);

        Collection<BsResyncItem> bsResyncItems = StreamEx.of(campaignIdsToResync)
                .map(cid -> new BsResyncItem(PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE, cid))
                .toList();
        long countOfResyncData = bsResyncService.addObjectsToResync(bsResyncItems);
        logger.info("shard: {}, added {} campaigns into bs_resync_queue table", shard, countOfResyncData);

        return updated.get();
    }

    CampaignsSelectionCriteria getCampaignsSelectionCriteria(Collection<Long> campaignIds) {
        return new CampaignsSelectionCriteria()
                .withCampaignTypes(singleton(CampaignType.PERFORMANCE))
                .withCampaignIds(campaignIds)
                .withStatusEmpty(false);
    }

    /**
     * Возвращает список кампаний с id in {@param campaignIds}, подходящих для миграции
     */
    abstract List<Campaign> getCampaignsForMigration(int shard, Collection<Long> campaignIds);

    /**
     * Изменение исходной стратегии с проверкой
     *
     * @return true - если измененная стратегия прошла валидацию
     */
    abstract boolean modifyAndValidateStrategy(int shard, Campaign campaign, DbStrategy strategyWas);

}
