package ru.yandex.direct.oneshot.oneshots.smartchangestrategy;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.oneshots.smartchangestrategy.repository.OneshotCampaignRepository;
import ru.yandex.direct.oneshot.oneshots.smartchangestrategy.validation.CampaignStrategyValidator;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER;

/**
 * Чистит AvgBid/FilterAvgBid у AUTOBUDGET_AVG_CPA_PER_* стратегий
 * <p>
 * Выполняется после {@link SmartChangeCpaStrategyToCpcOneshot}
 */
@Component
@Multilaunch
@Approvers({"ppalex"})
@ParametersAreNonnullByDefault
public class SmartCleanAvgBidInCpaStrategyIterativeOneshot extends SmartChangeStrategyBase {

    private static final Logger logger = LoggerFactory.getLogger(SmartCleanAvgBidInCpaStrategyIterativeOneshot.class);

    private final CampaignRepository campaignRepository;
    private final DslContextProvider dslContextProvider;
    private final CampaignStrategyValidator campaignStrategyValidator;

    @Autowired
    public SmartCleanAvgBidInCpaStrategyIterativeOneshot(CampaignRepository campaignRepository,
                                                         OneshotCampaignRepository oneshotCampaignRepository,
                                                         DslContextProvider dslContextProvider,
                                                         BsResyncService bsResyncService) {
        super(oneshotCampaignRepository, dslContextProvider, campaignRepository, bsResyncService, logger);
        this.campaignRepository = campaignRepository;
        this.dslContextProvider = dslContextProvider;
        campaignStrategyValidator = new CampaignStrategyValidator(logger);
    }

    @Override
    List<Campaign> getCampaignsForMigration(int shard, Collection<Long> campaignIds) {

        Collection<Campaign> campaigns = campaignRepository.getCampaigns(dslContextProvider.ppc(shard),
                getCampaignsSelectionCriteria(campaignIds), false);

        // Отбираем только стратегии AUTOBUDGET_AVG_CPA_PER_* c FilterAvgBid/AvgBid и FilterAvgCpa/AvgCpa
        return campaigns.stream()
                .filter(campaign -> {
                    StrategyData strategyData = campaign.getStrategy().getStrategyData();
                    if (!campaign.getStrategy().getStrategyName().equals(AUTOBUDGET_AVG_CPA_PER_FILTER)
                            && !campaign.getStrategy().getStrategyName().equals(AUTOBUDGET_AVG_CPA_PER_CAMP)) {
                        return false;
                    }
                    if (strategyData.getFilterAvgBid() == null && strategyData.getAvgBid() == null) {
                        return false;
                    }
                    return (strategyData.getFilterAvgCpa() != null && strategyData.getFilterAvgCpa().compareTo(BigDecimal.ZERO) > 0)
                            || (strategyData.getAvgCpa() != null && strategyData.getAvgCpa().compareTo(BigDecimal.ZERO) > 0);
                })
                .collect(Collectors.toList());
    }

    @Override
    boolean modifyAndValidateStrategy(int shard, Campaign campaign, DbStrategy strategyWas) {
        campaign.getStrategy().getStrategyData()
                .withAvgBid(null)
                .withFilterAvgBid(null);
        if (!campaignStrategyValidator.validateCampaignStrategy(shard, campaign)) {
            return false;
        }

        logger.info("shard: {}, cid: {}, update CAMPAIGNS.STRATEGY_DATA old: {} new: {}",
                shard, campaign.getId(), JsonUtils.toJson(strategyWas.getStrategyData()),
                JsonUtils.toJson(campaign.getStrategy().getStrategyData()));
        return true;
    }
}
