package ru.yandex.direct.oneshot.oneshots.smartchangestrategy.validation;


import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmartCampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.CampaignWithCustomStrategyValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.StrategyValidatorConstants;
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.StrategyValidatorConstantsBuilder;
import ru.yandex.direct.core.entity.campaign.service.validation.type.container.CampaignValidationContainer;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_PLATFORM_TYPE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.CAMPAIGN_TO_STRATEGY_TYPE;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@ParametersAreNonnullByDefault
public class CampaignStrategyValidator {

    private final Logger logger;

    public CampaignStrategyValidator(Logger logger) {
        this.logger = logger;
    }

    public boolean validateCampaignStrategy(int shard, Campaign campaign) {
        Currency currency = campaign.getCurrency().getCurrency();
        StrategyValidatorConstants constants =
                StrategyValidatorConstantsBuilder.build(campaign.getType(), currency);

        SmartCampaign smartCampaign = new SmartCampaign()
                .withType(campaign.getType())
                .withStrategy(campaign.getStrategy());

        // У autobudget_avg_cpc нет целей и платы за конверсии
        CampaignWithCustomStrategyValidator validator =
                new CampaignWithCustomStrategyValidator(currency,
                        emptySet(), null, null, null,
                        smartCampaign,
                        CAMPAIGN_TO_STRATEGY_TYPE.get(CampaignType.PERFORMANCE),
                        CAMPAIGN_TO_CONTEXT_STRATEGY_TYPE.get(CampaignType.PERFORMANCE),
                        CAMPAIGN_TO_PLATFORM_TYPE.get(CampaignType.PERFORMANCE),
                        constants, emptySet(),
                        CampaignValidationContainer.create(shard, null, null), null);

        ValidationResult<CampaignWithCustomStrategy, Defect> validationResult = validator.apply(smartCampaign);
        if (validationResult.hasAnyErrors()) {
            // Если цель не найдена в метрике - все равно меняем стратегию
            // Для данного кейса происходит только удаление avgBid
            if (hasOnlyGoalNotFoundError(validationResult)) {
                logger.warn("shard: {}, cid: {}, goal not found in Metrica :\n{}",
                        shard, campaign.getId(), StreamEx.of(validationResult.flattenErrors()).joining("\n\t"));
                return true;
            }
            logger.warn("shard: {}, cid: {}, modification of strategy caused the following errors:\n{}",
                    shard, campaign.getId(), StreamEx.of(validationResult.flattenErrors()).joining("\n\t"));
            return false;
        }
        return true;
    }

    /**
     * Возвращает true если есть только одна ошибка - не найдена цель в Метрике
     */
    private boolean hasOnlyGoalNotFoundError(ValidationResult<CampaignWithCustomStrategy, Defect> validationResult) {
        var flattenErrors = validationResult.flattenErrors();
        Path path = path(field(SmartCampaignWithCustomStrategy.STRATEGY), field(DbStrategy.STRATEGY_DATA),
                field(StrategyData.GOAL_ID));
        return flattenErrors.size() == 1
                && flattenErrors.get(0).getPath().equals(path)
                && flattenErrors.get(0).getDefect().equals(objectNotFound());
    }
}
