package ru.yandex.direct.oneshot.oneshots.strategy

import org.jooq.DSLContext
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.CampaignsMetatype
import ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.multipleConditionFilter
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereEqFilter
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereInFilter
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

/**
 * Oneshot на исправление стратегий в дочерних Смарт и ДО кампаниях ТК.
 * При создании проставили платформу BOTH, отчего страдает автобюджет, должно быть SEARCH для ДО и CONTEXT для смартов
 * Исправление в создании редактировании кампаний — DIRECT-167578
 */
@Component
@Approvers("ali-al", "kozobrodov", "buhter")
class UpdatePlatformForEcomCampaignsOneshot(
    private val campaignTypedRepository: CampaignTypedRepository,
    private val dslContextProvider: DslContextProvider,
) : ShardedOneshot<Void?, Void?> {
    companion object {
        private const val DEFAULT_UPDATE_CHUNK_SIZE = 500
    }

    override fun validate(inputData: Void?): ValidationResult<Void?, Defect<Any>> {
        return ValidationResult.success(inputData)
    }

    override fun execute(inputData: Void?, prevState: Void?, shard: Int): Void? {
        val dslContext = dslContextProvider.ppc(shard)
        val campaignsToUpdate = campaignTypedRepository.getTypedCampaigns(
            dslContext,
            multipleConditionFilter(
                whereInFilter(Tables.CAMPAIGNS.TYPE, listOf(CampaignsType.dynamic, CampaignsType.performance)),
                whereEqFilter(Tables.CAMPAIGNS.PLATFORM, CampaignsPlatform.both),
                whereEqFilter(Tables.CAMPAIGNS.METATYPE, CampaignsMetatype.ecom),
            )
        ).toList()
        val partitionedByType = campaignsToUpdate.partition { (it as CommonCampaign).type == CampaignType.DYNAMIC }

        partitionedByType.first.chunked(DEFAULT_UPDATE_CHUNK_SIZE).forEach {
            updatePlatformAndResetStatusBsSynced(dslContext, it, CampaignsPlatform.search)
        }

        partitionedByType.second.chunked(DEFAULT_UPDATE_CHUNK_SIZE).forEach {
            updatePlatformAndResetStatusBsSynced(dslContext, it, CampaignsPlatform.context)
        }

        return null
    }

    private fun updatePlatformAndResetStatusBsSynced(
        dslContext: DSLContext,
        campaigns: List<BaseCampaign>,
        newPlatform: CampaignsPlatform
    ) {
        val campaignIds = campaigns.map { campaign -> campaign.id }
        dslContext
            .update(Tables.CAMPAIGNS)
            .set(Tables.CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
            .set(Tables.CAMPAIGNS.PLATFORM, newPlatform)
            .where(Tables.CAMPAIGNS.CID.`in`(campaignIds))
            .execute()
    }
}
