package ru.yandex.direct.oneshot.oneshots.turnproviewforagencyclients;

import java.util.List;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.client.container.ClientsQueryFilter;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Ваншот для клиентов агентств включает отображение стратегий кампаний в про режиме
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers({"ssdmitriev", "pashkus"})
public class TurnStrategyProViewForAgencyClientsOneshot implements ShardedOneshot<Void,
        TurnStrategyProViewForAgencyClientsState> {
    private static final Logger logger = LoggerFactory.getLogger(TurnStrategyProViewForAgencyClientsOneshot.class);

    private static final int LIMIT = 5000;

    private final ClientRepository clientRepository;

    @Autowired
    public TurnStrategyProViewForAgencyClientsOneshot(ClientRepository clientRepository) {
        this.clientRepository = clientRepository;
    }

    @Nullable
    @Override
    public TurnStrategyProViewForAgencyClientsState execute(Void inputData,
                                                            TurnStrategyProViewForAgencyClientsState prevState,
                                                            int shard) {
        //если предыдущая стадия не инициализирована -- начали ваншот
        if (prevState == null) {
            logger.info("First iteration! shard: {}", shard);
            return new TurnStrategyProViewForAgencyClientsState(ClientId.fromLong(0));
        }
        //получаем 5000 следующих клиентов
        var limitedNextClientsFilter = ClientsQueryFilter.getLimitedNextClients(prevState.getLastClientId(), LIMIT);
        List<Client> clients = clientRepository.getClientsByFilter(shard, limitedNextClientsFilter);
        //если клиентов не оказалось, значит выгребли все данные
        if (clients.isEmpty()) {
            return null;
        }
        //отставляем только агентских клиентов с выключенным про режимом
        List<Client> clientsToUpdate = filterList(clients, this::isAgencyClientWithSimpleViewMode);
        //включаем им про режим
        turnOnProView(shard, clientsToUpdate);

        //берем последний clientId
        Long lastClientId = StreamEx.of(clients)
                .map(Client::getId)
                .max(Long::compare)
                .get();
        return new TurnStrategyProViewForAgencyClientsState(ClientId.fromLong(lastClientId));
    }

    private boolean isAgencyClientWithSimpleViewMode(Client client) {
        return client.getAgencyClientId() != null && !client.getIsProStrategyViewEnabled();
    }


    private void turnOnProView(int shard, List<Client> clients) {
        var changes = mapList(clients, client ->
                ModelChanges.build(client, Client.IS_PRO_STRATEGY_VIEW_ENABLED, true).applyTo(client));
        clientRepository.update(shard, changes);
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }

}
