package ru.yandex.direct.oneshot.oneshots.uc

import org.jooq.DSLContext
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.oneshots.uc.repository.OneshotUacCampaignRepository
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class DeleteGrutMissingUacCampaignsParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    val chunkSize: Int,
    val lastRow: Long = 0L
)

data class DeleteGrutMissingUacCampaignsState(
    var lastRow: Long = 0L,
    var deletedCount: Int = 0
) {
    fun increaseDeleteCount(addition: Int) {
        deletedCount += addition
    }

    fun withLastRow(row: Long): DeleteGrutMissingUacCampaignsState {
        this.lastRow = row
        return this
    }
}

/**
 * В ходе отказа GrUT 2022-04-08 ряд РМП и ТГО кампаний сохранились с ошибкой,
 * запись создалась в MySQL, но не создалась в GrUT.
 * Ваншот проходит по списку кампаний, удостоверяется, что это действительно ТГО/РМП кампании,
 * созданные через UAC и о них отсутствует запись в GrUT, а после удаляет такую кампанию.
 * Ваншот рассчитан на небольшое количество кампаний(до 2000) на запуск, логируются только cid'ы
 */
@Component
@Approvers("pavelkataykin", "khuzinazat", "bratgrim")
@Retries(5)
@Multilaunch
@PausedStatusOnFail
class DeleteGrutMissingUacCampaignsOneshot(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
    private val oneshotUacCampaignRepository: OneshotUacCampaignRepository,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val campaignService: CampaignService,
    private val clientRepository: ClientRepository,
) : SimpleOneshot<DeleteGrutMissingUacCampaignsParam, DeleteGrutMissingUacCampaignsState> {
    companion object {
        private val CID = YtField("cid", Long::class.javaObjectType)
        private const val CHUNK_LIMIT: Int = 300
    }

    private val logger = LoggerFactory.getLogger(this.javaClass)

    override fun validate(
        inputData: DeleteGrutMissingUacCampaignsParam
    ): ValidationResult<DeleteGrutMissingUacCampaignsParam, Defect<*>> {
        val validateBuilder = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(
            ytProvider,
            validateBuilder,
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(
        inputData: DeleteGrutMissingUacCampaignsParam,
        prevState: DeleteGrutMissingUacCampaignsState?
    ): DeleteGrutMissingUacCampaignsState? {
        val state = prevState ?: DeleteGrutMissingUacCampaignsState(inputData.lastRow)

        val chunkSize = maxOf(1, inputData.chunkSize)
        val startRow = state.lastRow
        val finishRow = chunkSize + startRow

        logger.info("Obtaining campaigns for delete from rows $startRow to $finishRow " +
            "(chunk size is ${chunkSize})")

        val campaignIdsByShard = getCampaignIdsByShardFromYt(inputData, startRow, finishRow)

        if (campaignIdsByShard.isNotEmpty()) {
            val deleted = deleteUacCampaignsMissingGrut(campaignIdsByShard)
            state.increaseDeleteCount(deleted)
        }
        return if (campaignIdsByShard.values.sumOf { it.size } < chunkSize) {
            logger.info("${state.deletedCount} were deleted in total")
            null
        } else {
            state.withLastRow(finishRow)
        }
    }

    private fun deleteUacCampaignsMissingGrut(campaignIdsByShard: Map<Int, List<Long>>): Int {
        var deletedCount = 0
        campaignIdsByShard.forEach { (shard, campaignIds) ->
            val dslContext = dslContextProvider.ppc(shard)
            val deleteIdListByClient = getGrutCampaignsAndCheckConditions(dslContext, campaignIds)
            val deleted = deleteBrokenCampaigns(shard, deleteIdListByClient)
            deletedCount += deleted
        }
        return deletedCount
    }

    private fun deleteBrokenCampaigns(
        shard: Int,
        campaignIdsByClient: Map<Long, List<Long>>
    ): Int {
        var deletedCount = 0
        campaignIdsByClient.forEach {
            val clientId = ClientId.fromLong(it.key)
            val client = clientRepository.get(shard, listOf(clientId)).first()
            val result = campaignService.deleteCampaigns(
                it.value,
                client.agencyUserId ?: client.chiefUid,
                clientId,
                Applicability.PARTIAL
            )
            val resultList = result.result.mapNotNull { res -> res.result }
            if (result.isSuccessful && it.value == resultList) {
                logger.info("${it.value.size} campaigns: ${it.value} were deleted " +
                    "for client: ${it.key}")
                deletedCount += it.value.size
            } else {
                if (resultList.isNotEmpty()) {
                    logger.info("${resultList.size} campaigns: $resultList were deleted for client: ${it.key}")
                    deletedCount += it.value.size
                }
                val failedToDelete = it.value.minus(resultList.toSet())
                logger.info("${failedToDelete.size} campaigns: $failedToDelete has not been deleted " +
                    "for client: ${it.key}")
            }
        }
        return deletedCount
    }

    private fun getGrutCampaignsAndCheckConditions(
        dslContext: DSLContext,
        campaignIds: List<Long>
    ): Map<Long, List<Long>> {
        return oneshotUacCampaignRepository
            .getUacCampaignsByClientId(dslContext, campaignIds)
            .mapValues { getNonexistentInGrutCampaigns(it.value) }
            .filterValues { it.isNotEmpty() }
    }

    private fun getNonexistentInGrutCampaigns(campaignIds: List<Long>): List<Long> {
        val existsInGrutCampaignIds = mutableSetOf<Long>()
        campaignIds
            .chunked(CHUNK_LIMIT)
            .forEach { ids ->
                val result = grutUacCampaignService.getCampaigns(ids.map { it.toString() })
                if (result.isNotEmpty()) {
                    val grutCampaignIds = result
                        .map { it.meta.id }
                    existsInGrutCampaignIds.addAll(grutCampaignIds)
                    logger.info("${result.size} campaigns of ${campaignIds.size} were found in GrUT. " +
                        "Removing $grutCampaignIds from delete list")
                }
            }
        return campaignIds.minus(existsInGrutCampaignIds)
    }

    private fun getCampaignIdsByShardFromYt(
        inputData: DeleteGrutMissingUacCampaignsParam,
        startRow: Long, finishRow: Long
    ): Map<Int, List<Long>> {
        val campaignIds = mutableListOf<Long>()
        ytProvider.getOperator(inputData.ytCluster).readTableByRowRange(
            YtTable(inputData.tablePath),
            { campaignIds.add(it.valueOf(CID)) },
            YtTableRow(listOf(CID)),
            startRow,
            finishRow
        )
        logger.info("${campaignIds.size} campaigns were obtained for delete")
        return shardHelper.groupByShard(campaignIds, ShardKey.CID).shardedDataMap
    }
}
