package ru.yandex.direct.oneshot.oneshots.uc

import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toImageAsset
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toEpochSecond
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.service.AssetIdToThumb
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.entity.uac.service.GrutUacContentService
import ru.yandex.direct.core.entity.uac.service.GrutUacDeleteImageContentService
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import ru.yandex.grut.objects.proto.MediaType

data class DeleteUnusedAssetsForClientParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    val chunkSize: Long? = null,
    val maxLookupTime: LocalDateTime?,
    val lastRow: Long = 0L,
    val deleteLimit: Long = 5000L,
    val idleTime: Long = 500L
)

data class DeleteUnusedAssetsForClientState(
    var lastRow: Long = 0L,
    var deletedCount: Long = 0L
) {
    fun withLastRow(row: Long): DeleteUnusedAssetsForClientState {
        this.lastRow = row
        return this
    }
}

data class DeleteCandidateAsset(
    val clientId: String,
    val assetId: String,
)

/**
 * Производительность DeleteUnusedAssetsFromMDSJob на данный момент не позволяет
 * удалять неиспользуемые ассеты с требуемой скоростью.
 * В качестве вспомогательной меры создан этот ваншот. Он позволяет проверить все ассеты на предмет использования
 * в кампаниях и удалить неиспользуемые созданные позднее maxLookupTime
 */
@Component
@Approvers("pavelkataykin", "khuzinazat", "bratgrim")
@Retries(5)
@Multilaunch
@PausedStatusOnFail
class DeleteUnusedAssetsForClientOneshot(
    private val ytProvider: YtProvider,
    private val grutUacDeleteImageContentService: GrutUacDeleteImageContentService,
    private val grutUacContentService: GrutUacContentService,
    private val grutUacCampaignService: GrutUacCampaignService,
) : SimpleOneshot<DeleteUnusedAssetsForClientParam, DeleteUnusedAssetsForClientState> {
    companion object {
        private val CLIENT_ID = YtField("client_id", String::class.javaObjectType)
        private val ASSET_ID = YtField("asset_id", String::class.javaObjectType)
        private const val DELIMITER = "/"
        private const val LIMIT = 4000
        private const val YT_DEFAULT_CHUNK_SIZE = 1000L
    }

    private val logger = LoggerFactory.getLogger(this.javaClass)

    override fun validate(
        inputData: DeleteUnusedAssetsForClientParam
    ): ValidationResult<DeleteUnusedAssetsForClientParam, Defect<*>> {
        val validationBuilder = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(
            ytProvider,
            validationBuilder,
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(
        inputData: DeleteUnusedAssetsForClientParam,
        prevState: DeleteUnusedAssetsForClientState?
    ): DeleteUnusedAssetsForClientState? {
        val state = prevState ?: DeleteUnusedAssetsForClientState(inputData.lastRow)

        val chunkSize = minOf(inputData.chunkSize ?: YT_DEFAULT_CHUNK_SIZE, LIMIT.toLong())
        val startRow = state.lastRow
        val finishRow = chunkSize + startRow

        logger.info("Obtaining clients, rows #$startRow to #$finishRow")

        val deleteCandidateAssetsByClientId = getDeleteCandidateAssets(inputData, startRow, finishRow)

        deleteCandidateAssetsByClientId.forEach { (clientId, assets) ->
            state.deletedCount = deleteUnusedImageAssets(
                clientId,
                assets.map { it.assetId },
                maxOf(1, inputData.deleteLimit),
                inputData.maxLookupTime,
                maxOf(1, inputData.idleTime),
                state.deletedCount,
            )
            if (state.deletedCount >= inputData.deleteLimit) {
                logger.info("The delete limit has been reached on clientId $clientId." +
                    " ${state.deletedCount} were deleted in total")
                return null
            }
        }

        return if (deleteCandidateAssetsByClientId.values.sumOf { it.size } < chunkSize) {
            logger.info("${state.deletedCount} assets were deleted in total")
            null
        } else {
            state.withLastRow(finishRow)
        }
    }

    private fun deleteUnusedImageAssets(
        clientId: String,
        assets: List<String>,
        deleteLimit: Long,
        maxLookupTime: LocalDateTime?,
        idleTime: Long,
        totallyDeleted: Long = 0,
    ): Long {
        var deleted = totallyDeleted
        val result = grutUacContentService.getAssets(assets)
            .asSequence()
            .filter {
                it.meta.mediaType == MediaType.EMediaType.MT_IMAGE
                    && (maxLookupTime == null || toEpochSecond(maxLookupTime) * 1_000_000 > it.meta.creationTime)
            }
            .map { AssetIdToThumb(it.meta.id.toIdString(), it.toImageAsset().mdsInfo.thumb) }
            .toSet()

        if (result.isEmpty()) return deleted

        val usedAssetIds = grutUacCampaignService.getClientCampaignAssets(
            clientId,
            result
                .asSequence()
                .map { it.id }
                .toSet(),
            LIMIT.toLong()
        )

        result.asSequence()
            .filterNot { it.id in usedAssetIds }
            .forEach {
                if (deleteAsset(it, idleTime) && ++deleted >= deleteLimit) {
                    logger.info("The delete limit has been reached. Aborting operation")
                    return deleted
                }
            }

        Thread.sleep(idleTime)

        return deleted
    }

    private fun deleteAsset(
        assetIdToThumb: AssetIdToThumb,
        idleTime: Long,
    ): Boolean {
        val avatarId = grutUacDeleteImageContentService.uacThumbToAvatarId(assetIdToThumb.thumb, DELIMITER)
        if (avatarId != null) {
            return grutUacDeleteImageContentService.deleteUnusedAsset(assetIdToThumb.id, avatarId, idleTime)
        }
        return false
    }

    private fun getDeleteCandidateAssets(
        inputData: DeleteUnusedAssetsForClientParam,
        startRow: Long,
        finishRow: Long,
    ): Map<String, List<DeleteCandidateAsset>> {
        val deleteCandidateAssets = mutableListOf<DeleteCandidateAsset>()
        ytProvider.getOperator(inputData.ytCluster).readTableByRowRange(
            YtTable(inputData.tablePath),
            {
                deleteCandidateAssets.add(DeleteCandidateAsset(it.valueOf(CLIENT_ID), it.valueOf(ASSET_ID)))
            },
            YtTableRow(listOf(CLIENT_ID, ASSET_ID)),
            startRow,
            finishRow
        )
        logger.info("${deleteCandidateAssets.size} assets were obtained for the analysis")
        return deleteCandidateAssets.groupBy { it.clientId }
    }
}
