package ru.yandex.direct.oneshot.oneshots.uc

import com.google.common.collect.Lists
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.oneshot.oneshots.uc.repository.OneshotUacBannerRepository
import ru.yandex.direct.oneshot.oneshots.uc.repository.OneshotUacCampaignRepository
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.State
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class InputData(
    val clientId: Long?
)

/**
 * Ваншот для сброса флага показа 'statusShow' у архивных баннеров uac ТГО кампаний
 */
@Component
@Multilaunch
@Approvers("mspirit", "dimitrovsd", "khuzinazat", "a-dubov")
@Retries(5)
@PausedStatusOnFail
class DisableShowStatusForArchivedBannersOneshot @Autowired constructor(
    private val oneshotCampaignRepository: OneshotUacCampaignRepository,
    private val oneshotBannerRepository: OneshotUacBannerRepository,
    private val shardHelper: ShardHelper,
) : ShardedOneshot<InputData, State?> {
    companion object {
        private val logger = LoggerFactory.getLogger(DisableShowStatusForArchivedBannersOneshot::class.java)

        // Макс. количество проблемных баннеров на кампанию = 99 -> для 20 кампаний отберется не более 1980 баннеров
        private const val CAMPAIGN_CHUNK_SIZE = 20
    }

    override fun validate(
        inputData: InputData?
    ): ValidationResult<InputData?, Defect<*>> {
        if (inputData?.clientId == null) {
            return ValidationResult.success(inputData)
        }

        val builder: ItemValidationBuilder<InputData, Defect<*>> = ItemValidationBuilder.of(inputData)
        builder.item(inputData.clientId, "clientId")
            .check(NumberConstraints.greaterThan(0L))
            .check(Constraint.fromPredicate(
                { clientId -> isClientExistAndHisCampaignsAreSuitable(clientId) }, CommonDefects.objectNotFound()
            ), When.isValid())
        return builder.result
    }

    override fun execute(
        inputData: InputData?,
        prevState: State?,
        shard: Int
    ): State? {
        val campaignIds = oneshotCampaignRepository
            .getUacCampaignIdsWithDefectArchivedBanners(shard, inputData?.clientId)
        logger.info("shard $shard: got ${campaignIds.size} campaigns that have problems with archived banners")

        var updatedBannersCount = 0
        for (campaignIdsChunk in Lists.partition(campaignIds, CAMPAIGN_CHUNK_SIZE)) {
            logger.info("shard $shard: got campaigns $campaignIdsChunk to fix problems with archived banners")

            val fixedBanners = oneshotBannerRepository
                .fixDefectArchivedBannersForCampaigns(shard, campaignIdsChunk)
            updatedBannersCount += fixedBanners
        }
        logger.info("shard $shard: $updatedBannersCount banners have been fixed in total")
        return null
    }

    private fun isClientExistAndHisCampaignsAreSuitable(clientId: Long): Boolean {
        val shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId))
        if (shard == ShardSupport.NO_SHARD) {
            return false
        }

        val campaignIds = oneshotCampaignRepository.getUacCampaignIdsWithDefectArchivedBanners(shard, clientId)
        return campaignIds.isNotEmpty()
    }
}
