package ru.yandex.direct.oneshot.oneshots.uc

import java.time.Instant
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.mobilecontent.model.OsType
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.entity.uac.service.appinfo.ParseAppStoreUrlService
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.base.YtState
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.client.Schema

data class FixImpressionInputData(
    val ytCluster: YtCluster,
    val tablePath: String,
    val chunk: Int = 1,
    val lastRow: Long = 0L
)

/**
 * Ваншот проверяет список заявок на основе таблицы YT из входных параметров на предмет целостности VTA-ссылки,
 * если ссылка битая, то ваншот попытается сгенерировать ссылку на основе данных из кликовой ссылки, если это возможно,
 * далее поврежденные данные будут заменены либо на верную VTA ссылку, либо будут занулены.
 *
 *  Пример входных данных: {"ytCluster": "HAHN", "tablePath": "//home/direct/test/table", "chunk": 1, "lastRow": 0},
 *  где таблица из tablePath должна содержать одну колонку с именем "cid", которая содержит id кампаний для проверки.
 */
@Component
@Approvers("pavelkataykin", "khuzinazat", "bratgrim")
@Retries(5)
@Multilaunch
@PausedStatusOnFail
class FixNullTrackerIdInImpressionUrlOneshot(
    private val trackingUrlParseService: TrackingUrlParseService,
    ytProvider: YtProvider,
    parseAppStoreUrlService: ParseAppStoreUrlService,
    grutApiService: GrutApiService,
    ydbGrutConverterYtRepository: YdbGrutConverterYtRepository,
    uacBannerService: UacBannerService,
    shardHelper: ShardHelper,
    clientRepository: ClientRepository,
    grutTransactionProvider: GrutTransactionProvider,
) : BaseFixBrokenImpressionUrlOneshot(
    ytProvider,
    trackingUrlParseService,
    parseAppStoreUrlService,
    grutApiService,
    ydbGrutConverterYtRepository,
    uacBannerService,
    shardHelper,
    clientRepository,
    grutTransactionProvider,
) {
    override fun impressionHasError(impressionUrl: String?, trackingUrl: String?): Boolean {
        if (impressionUrl == null) return false
        val error = trackingUrlParseService.validateImpression(impressionUrl)
        return (error != null || impressionUrl.contains("/null"))
    }
}

abstract class BaseFixBrokenImpressionUrlOneshot(
    private val ytProvider: YtProvider,
    private val trackingUrlParseService: TrackingUrlParseService,
    private val parseAppStoreUrlService: ParseAppStoreUrlService,
    private val grutApiService: GrutApiService,
    private val ydbGrutConverterYtRepository: YdbGrutConverterYtRepository,
    private val uacBannerService: UacBannerService,
    private val shardHelper: ShardHelper,
    private val clientRepository: ClientRepository,
    private val grutTransactionProvider: GrutTransactionProvider,
) : SimpleOneshot<FixImpressionInputData, YtState> {
    companion object {
        private const val RETRY_COUNT = 3
    }

    private val logger = LoggerFactory.getLogger(this.javaClass)

    override fun validate(inputData: FixImpressionInputData): ValidationResult<FixImpressionInputData, Defect<*>> {
        val validationBuilder = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(
            ytProvider,
            validationBuilder,
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(inputData: FixImpressionInputData, prevState: YtState?): YtState? {
        val state = prevState ?: YtState()

        val chunkSize = maxOf(1, inputData.chunk)
        val startRow = prevState?.nextRow ?: inputData.lastRow
        val finishRow = chunkSize + startRow

        val cids = getCampaignIDs(inputData, startRow, finishRow)
        if (cids.isEmpty()) return null

        cids.forEach { id ->
            grutTransactionProvider.runRetryable(RETRY_COUNT) {
                val campaign = grutApiService.briefGrutApi.getBrief(id.toIdLong())
                if (campaign != null) {
                    val trackingURL = campaign.spec.campaignBrief.targetHref.trackingUrl
                    val impressionURL = campaign.spec.campaignBrief.targetHref.impressionUrl
                    if (impressionHasError(impressionURL, trackingURL)) {
                        val campaignId = campaign.meta.id
                        logger.info("Campaign $campaignId contains broken VTA link: $impressionURL")
                        val newUrl = newImpression(trackingURL)
                        updateImpression(campaignId, campaign.spec.campaignBrief.targetHref, newUrl)

                        val clientId = ClientId.fromLong(campaign.meta.clientId)
                        val shard = shardHelper.getShardByClientId(clientId)
                        val client = clientRepository.get(shard, listOf(clientId)).first()
                        uacBannerService.updateAdsDeferred(
                            clientId,
                            client.agencyUserId ?: client.chiefUid,
                            campaignId.toIdString())
                        logger.info("Campaign $campaignId has been queued for ads update")
                    }
                }
            }
        }
        state.nextRow = finishRow
        return state
    }

    private fun updateImpression(
        campaignId: Long,
        targetHref: Campaign.TCampaignBrief.TTargetHref,
        newUrl: String?
    ) {
        val newTargetHref = Campaign.TCampaignBrief.TTargetHref.newBuilder().mergeFrom(targetHref)

        if (newUrl == null) newTargetHref.clearImpressionUrl()
        else newTargetHref.impressionUrl = newUrl

        grutApiService.briefGrutApi.updateBrief(
            Schema.TCampaign.newBuilder().apply {
                metaBuilder.apply {
                    id = campaignId
                }
                specBuilder.apply {
                    updateTime = Instant.now().epochSecond.toInt()
                    campaignBriefBuilder.apply {
                        this.briefSynced = false
                        this.targetHref = newTargetHref.build()
                    }.build()
                }
            }.build(),
            setPaths = listOf(
                "/spec/update_time",
                "/spec/campaign_brief/brief_synced",
                "/spec/campaign_brief/target_href"
            ),
        )
        logger.info("Setting VTA link to $newUrl for campaign: $campaignId")
    }

    private fun newImpression(trackingUrl: String): String? {
        val result = parseAppStoreUrlService.getAppStoreUrl(trackingUrl)
        val osType = (if (result.hasAnyErrors()) null else result.value.parsedUrl.osType) ?: return null

        val platform = if (osType == OsType.IOS) Platform.IOS else Platform.ANDROID
        val response = trackingUrlParseService.createImpressionUrl(trackingUrl, platform)
        return response.second?.getUrl()
    }

    abstract fun impressionHasError(impressionUrl: String?, trackingUrl: String? = null): Boolean

    private fun getCampaignIDs(
        inputData: FixImpressionInputData,
        startRow: Long,
        finishRow: Long,
    ): List<String> = ydbGrutConverterYtRepository.getCampaignIdsFromYtTable(
        inputData.ytCluster,
        inputData.tablePath,
        startRow,
        finishRow
    ).map { it.toString() }
}
