package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.grut.api.BriefBanner
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.Environment
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.GrutUacConverterRepository
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.MigrationResult
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.grut.objects.proto.Banner

data class BannersCreateParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    val dryRun: Boolean,
    val writeResults: Boolean?
)

@Component
@Approvers("mspirit", "dimitrovsd", "elwood", "pavelkataykin")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class GrutYdbBannersMigrateOneshot(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val uacYdbDirectAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val adGroupRepository: AdGroupRepository,
    private val shardHelper: ShardHelper,
    private val grutUacConverterRepository: GrutUacConverterRepository,
    private val grutApiService: GrutApiService,
) : SimpleOneshot<BannersCreateParam, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutYdbBannersMigrateOneshot::class.java)
        private val RESULT_TABLE_PATH = "//home/direct/test/uacconverter/GrutYdbBannersMigrateResult_${Environment.getCached().name.lowercase()}"
    }

    override fun validate(inputData: BannersCreateParam) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }


    override fun execute(inputData: BannersCreateParam, prevState: State?): State? {
        logger.info("Start from state=$prevState")
        val chunkSize = 1
        uacConverterYtRepository.createResultTableIfNotExists(RESULT_TABLE_PATH)
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val adGroupIds = uacConverterYtRepository.getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        if (adGroupIds.isEmpty()) return null
        val adGroupId = adGroupIds[0]
        try {
            adGroupId.let {
                val uacAdGroup = uacYdbDirectAdGroupRepository.getDirectAdGroupByDirectAdGroupId(it)
                if (uacAdGroup == null) {
                    val message = "AdGroup $it not fount in ydb"
                    logger.warn(message)
                    val migrationResult = MigrationResult(it, false, message)
                    uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                    return@let
                }
                logger.info("Start handle adGroup ${uacAdGroup.directAdGroupId} ydb id ${uacAdGroup.id}")
                val shard = shardHelper.getShardByGroupId(uacAdGroup.directAdGroupId)
                val directCampaignId = adGroupRepository.getCampaignIdsByAdGroupIds(shard, listOf(uacAdGroup.directAdGroupId))[uacAdGroup.directAdGroupId]
                logger.info("Direct campaign id $directCampaignId for adGroup id ${uacAdGroup.directAdGroupId}")
                val grutCampaign = getGrutCampaign(directCampaignId!!)
                if (grutCampaign == null) {
                    val message = "Campaign $directCampaignId for adGroup id ${uacAdGroup.directAdGroupId} not found in grut, not migrate"
                    logger.warn(message)
                    val migrationResult = MigrationResult(uacAdGroup.directAdGroupId, false, message)
                    uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                    return@let
                }
                logger.info("Campaign $directCampaignId for adGroup id ${uacAdGroup.directAdGroupId} found in grut, not migrate")
                val banners = getYdbBanners(uacAdGroup.id)
                val grutBannerIds = grutApiService.briefBannerGrutApi.selectBanners(
                    filter = "[/meta/campaign_id] = ${grutCampaign.meta.id}",
                    index = "banners_by_campaign",
                    attributeSelector = listOf("/meta/id", "/meta/campaign_id")
                )
                    .filter { grutBanner -> grutBanner.meta.campaignId == directCampaignId }
                    .map { grutBanner -> grutBanner.meta.id }
                    .toSet()
                if (grutBannerIds.size > 1) {
                    val message = "Campaign $directCampaignId for adGroup id ${uacAdGroup.directAdGroupId} was changed after migrate, it has ${grutBannerIds.size} banners"
                    logger.error(message)
                    val migrationResult = MigrationResult(uacAdGroup.directAdGroupId, false, message)
                    uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                    return@let
                }

                val bannersToCreate = banners.filterNot { banner -> grutBannerIds.contains(banner.directAdId) }
                logger.info("Banners for campaign $directCampaignId for adGroup ${uacAdGroup.directAdGroupId} to create $bannersToCreate")
                if (inputData.dryRun) {
                    logger.info("Do not really create banners, there is dry run mode")
                    return@let
                }
                if (bannersToCreate.isEmpty()) {
                    val migrationResult = MigrationResult(uacAdGroup.directAdGroupId, true, "No banners to migrate")
                    uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                    return@let
                }
                val grutBanners = bannersToCreate
                    .map { banner ->
                        getBanner(directCampaignId, uacAdGroup.directAdGroupId, banner)
                    }
                grutUacConverterRepository.createBanners(directCampaignId, grutBanners)
                val migrationResult = MigrationResult(uacAdGroup.directAdGroupId, true, null)
                uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
            }
        } catch (e: Exception) {
            logger.error("Got exception", e)
            val migrationResult = MigrationResult(adGroupIds[0], false, e.message)
            uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
        }

        return State(lastRow)
    }

    private fun getYdbBanners(ydbAdGroupId: String): List<UacYdbDirectAd> {
        val allYdbDirectAds = mutableListOf<UacYdbDirectAd>()

        var fromUacAdId = 0L
        while (true) {
            val ydbDirectAdsFromYdb = uacYdbDirectAdRepository
                .getByDirectAdGroupId(listOf(ydbAdGroupId), fromUacAdId, 990L)

            allYdbDirectAds.addAll(ydbDirectAdsFromYdb)
            if (ydbDirectAdsFromYdb.size < 990L) {
                break
            }

            // Здесь нельзя сравнивать и брать наибольший, т.к. значения id выше long.max - отрицательные
            fromUacAdId = ydbDirectAdsFromYdb.last().id.toIdLong()
        }
        return allYdbDirectAds
    }


    private fun getGrutCampaign(campaignId: Long) =
        grutApiService.briefGrutApi.getBrief(campaignId)

    private fun getBanner(directCampaignId: Long, directAdGroupId: Long, ad: UacYdbDirectAd) =
        BriefBanner(
            id = ad.directAdId!!,
            adGroupId = directAdGroupId,
            briefId = directCampaignId,
            source = Banner.EBannerSource.BS_DIRECT,
            assetIds = listOfNotNull(
                ad.textContentId,
                ad.titleContentId,
                ad.directVideoContentId,
                ad.directImageContentId,
                ad.directHtml5ContentId,
                ad.directContentId,
            ).map { it.toIdLong() },
            assetLinksIds = listOf(),
            status = ad.status.toEBannerStatus()
        )

    private fun DirectAdStatus.toEBannerStatus(): Banner.TBannerSpec.EBannerStatus {
        return when (this) {
            DirectAdStatus.CREATED -> Banner.TBannerSpec.EBannerStatus.BSS_CREATED
            DirectAdStatus.ERROR_UNKNOWN -> Banner.TBannerSpec.EBannerStatus.BSS_ERROR_UNKNOWN
            DirectAdStatus.MODERATING -> Banner.TBannerSpec.EBannerStatus.BSS_MODERATING
            DirectAdStatus.ACTIVE -> Banner.TBannerSpec.EBannerStatus.BSS_ACTIVE
            DirectAdStatus.REJECTED -> Banner.TBannerSpec.EBannerStatus.BSS_REJECTED
            DirectAdStatus.ARCHIVED -> Banner.TBannerSpec.EBannerStatus.BSS_ARCHIVED
            DirectAdStatus.DELETED -> Banner.TBannerSpec.EBannerStatus.BSS_DELETED
            else -> Banner.TBannerSpec.EBannerStatus.BSS_NOT_SPECIFIED
        }
    }
}
