package ru.yandex.direct.oneshot.oneshots.uc

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.forecast.CpaEstimatesRepository
import ru.yandex.direct.core.entity.forecast.model.CpaEstimate
import ru.yandex.direct.core.entity.forecast.model.CpaEstimateAttributionModel
import ru.yandex.direct.core.entity.forecast.model.CpaEstimateGoalType
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

private val ATTRIBUTION_MODELS = CpaEstimateAttributionModel.values()

@Component
@Multilaunch
@Approvers("gerdler", "dimitrovsd")
class PullCpaEstimatesTableOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val cpaEstimatesRepository: CpaEstimatesRepository
) : SimpleOneshot<Void, Void> {

    companion object {
        const val TABLE_PATH = "//home/direct/tmp/dimitrovsd/cpa_estimates"
        const val CHUNK_SIZE = 1000
    }

    override fun validate(inputData: Void?): ValidationResult<Void, Defect<Any>> {
        return ValidationResult.success(inputData)
    }

    override fun execute(inputData: Void?, prevState: Void?): Void? {
        val cpaEstimatesFromYt = mutableListOf<CpaEstimate>()

        ytProvider.getOperator(YtCluster.HAHN)
            .readTable(YtTable(TABLE_PATH), CpaEstimatesYtTableRow(), { cpaEstimatesFromYt.add(mapRowToModel(it)) })

        cpaEstimatesFromYt.chunked(CHUNK_SIZE).forEach {
            cpaEstimatesRepository.addOrUpdate(it)
        }

        return null
    }
}

private fun mapRowToModel(ytRow: CpaEstimatesYtTableRow): CpaEstimate {
    return CpaEstimate().apply {
        businessCategory = ytRow.category
        regionId = ytRow.regionId
        medianCpa = ytRow.medianCpa.toBigDecimal()
        minCpa = ytRow.minCpa.toBigDecimal()
        maxCpa = ytRow.maxCpa.toBigDecimal()
        goalType = ytRow.goalType.toGoalType()
        attributionModel = ytRow.attributionType.toAttributionType()
    }
}

private fun String.toGoalType(): CpaEstimateGoalType {
    return CpaEstimateGoalType.valueOf(this.toUpperCase())
}

private fun Long.toAttributionType(): CpaEstimateAttributionModel {
    return ATTRIBUTION_MODELS[this.toInt() - 1]
}

class CpaEstimatesYtTableRow
    : YtTableRow(listOf(ATTRIBUTION_TYPE, CATEGORY, REGION_ID, GOAL_TYPE, MAX_CPA, MEDIAN_CPA, MIN_CPA)) {

    val attributionType: Long
        get() = valueOf(ATTRIBUTION_TYPE)
    val category: String
        get() = valueOf(CATEGORY)
    val regionId: Long
        get() = valueOf(REGION_ID)
    val goalType: String
        get() = valueOf(GOAL_TYPE)
    val maxCpa: Double
        get() = valueOf(MAX_CPA)
    val medianCpa: Double
        get() = valueOf(MEDIAN_CPA)
    val minCpa: Double
        get() = valueOf(MIN_CPA)

    companion object {
        private val ATTRIBUTION_TYPE = YtField("AttributionType", Long::class.java)
        private val CATEGORY = YtField("Category", String::class.java)
        private val REGION_ID = YtField("RegionID", Long::class.java)
        private val GOAL_TYPE = YtField("goal_type", String::class.java)
        private val MAX_CPA = YtField("max_cpa", Double::class.java)
        private val MEDIAN_CPA = YtField("median_cpa", Double::class.java)
        private val MIN_CPA = YtField("min_cpa", Double::class.java)
    }
}
