package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService
import ru.yandex.direct.core.entity.campaign.converter.MobileContentConverter.altAppStoresToDb
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAlternativeStore
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MOBILE_CONTENT
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class UacAltAppStoresParam(
    val ytCluster: YtCluster,
    val tablePath: String,
)

data class UacAltAppStoresUpdateState(
    val lastId: Long = 0L
)

/**
 * Ваншот для проставления для всех кампаний из whitelist флаги про appgallerey и getapps = true (altAppStores)
 * Лист с приложениями с whitelist https://yt.yandex-team.ru/hahn/navigation?path=//home/rmp/production/deeplinks/appget/latest/links
 * <p>
 * Алгоритм ваншота:
 * - достаем appId лист с whitelist таблички (сейчас там 46 приложений)
 * - достаем mobileAppId лист с таблички mobile_apps
 * - фильтруем mobileAppId лист, если у mobile_app есть приложение с уайтлиста, то берем
 * - обновляем табличку campaigns_mobile_content, ставим флаги про appgallerey и getapps = true сторам
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Retries(2)
@Approvers("pavelkataykin", "khuzinazat", "mspirit", "dlyange", "bratgrim")
class SetAltAppStoresToWhitelistCampaignsOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val mobileAppRepository: MobileAppRepository,
    private val mobileContentRepository: MobileContentRepository,
    private val bsResyncService: BsResyncService,
    private val bannerRelationsRepository: BannerRelationsRepository
) : ShardedOneshot<UacAltAppStoresParam, UacAltAppStoresUpdateState?> {

    companion object {
        private val logger = LoggerFactory.getLogger(this::class.java)
        private const val CHUNK_SIZE = 1000L
        private val ALTERNATIVE_APP_STORES = altAppStoresToDb(
            setOf(
                MobileAppAlternativeStore.XIAOMI_GET_APPS,
                MobileAppAlternativeStore.HUAWEI_APP_GALLERY
            )
        )

        private val DOWNLOAD_DEEPLINK = YtField("downloadDeeplink", String::class.javaObjectType)
        private val REGION_NAME = YtField("regionName", String::class.javaObjectType)
        private val BUNDLE_ID = YtField("bundleId", String::class.javaObjectType)
    }

    override fun validate(
        inputData: UacAltAppStoresParam
    ): ValidationResult<UacAltAppStoresParam, Defect<*>> {
        val validateBuilder = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(
            ytProvider,
            validateBuilder,
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(
        inputData: UacAltAppStoresParam,
        prevState: UacAltAppStoresUpdateState?,
        shard: Int
    ): UacAltAppStoresUpdateState? {
        val startId = prevState?.lastId ?: 0L
        logger.info("Shard=$shard: Start from row=$startId")

        // read apks from whitelist
        val apks = readInputTable(inputData).toSet()
        logger.info("got ${apks.size} apks from xiaomi and huawei whitelist")

        // read mobile contents by chunk
        val mobileContents = mobileContentRepository.getMobileContentsChunk(shard, startId, CHUNK_SIZE)

        // filter mobile contents by apks from whitelist, return list of mobile_content_id
        val mobileContentsByAppId = mobileContents
            .filter { apks.contains(it.storeContentId) || (!it.bundleId.isNullOrEmpty() && apks.contains(it.bundleId)) }
            .mapNotNull { it.id }
        logger.info("got ${mobileContentsByAppId.size} filtered mobile content ids by apks")

        // get mobileAppIds by mobileContentIds
        val mobileAppIds = mobileAppRepository.getMobileAppIdsByMobileContentIds(shard, mobileContentsByAppId)
        logger.info("got ${mobileAppIds.size} mobile apps ids, at $shard shard")

        if (mobileAppIds.isNotEmpty()) {
            updateCampaignsMobileContentAltAppStores(
                shard,
                mobileAppIds,
            )
        }

        if (mobileContents.size < CHUNK_SIZE) {
            logger.info("Shard=$shard: Last iteration finished")
            return null
        }
        return UacAltAppStoresUpdateState(mobileContents.last().id)
    }

    private fun updateCampaignsMobileContentAltAppStores(
        shard: Int,
        mobileAppIds: Collection<Long>,
    ) {
        val oldAltAppStores = getAlternativeAppStores(shard, mobileAppIds)
        logger.info("Old alternative stores with size ${oldAltAppStores.size}")
        oldAltAppStores.forEach { idsToStores ->
            logger.info("Old alternative stores with CID: ${idsToStores.key} and altAppStores: ${idsToStores.value}")
        }
        val campaignIdsToUpdate = oldAltAppStores.keys.toList()

        updateCampaignsMobileContent(shard, campaignIdsToUpdate)
        logger.info("Update campaigns_mobile_content, set $ALTERNATIVE_APP_STORES for ${campaignIdsToUpdate.size} campaignIdsToUpdate")

        val nonArchivedBannerIdsByCampaignIds =
            bannerRelationsRepository.getCampaignIdByNonArchivedBannerId(shard, campaignIdsToUpdate)
        val bannerIds = nonArchivedBannerIdsByCampaignIds.keys.toList()

        val bsResyncItems = bannerIds
            .map {
                BsResyncItem(
                    BsResyncPriority.DEFAULT,
                    nonArchivedBannerIdsByCampaignIds[it]!!,
                    it,
                    null
                )
            }
        val addedForResync = bsResyncService.addObjectsToResync(bsResyncItems)
        logger.info("$addedForResync campaigns has been queued for bs_resync_queue table")
    }

    private fun updateCampaignsMobileContent(shard: Int, campaignIdsToUpdate: Collection<Long>) {
        dslContextProvider.ppc(shard)
            .update(CAMPAIGNS_MOBILE_CONTENT)
            .set(CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES, ALTERNATIVE_APP_STORES)
            .where(CAMPAIGNS_MOBILE_CONTENT.CID.`in`(campaignIdsToUpdate))
            .execute()
    }

    private fun getAlternativeAppStores(shard: Int, mobileAppIds: Collection<Long>): Map<Long, String> {
        return dslContextProvider.ppc(shard)
            .select(CAMPAIGNS_MOBILE_CONTENT.CID, CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES)
            .from(CAMPAIGNS_MOBILE_CONTENT)
            .where(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID.`in`(mobileAppIds))
            .and(CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES.isNull
                .or(CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES.length().eq(0))
            )
            .fetchMap(CAMPAIGNS_MOBILE_CONTENT.CID, CAMPAIGNS_MOBILE_CONTENT.ALTERNATIVE_APP_STORES)
    }

    private fun readInputTable(inputData: UacAltAppStoresParam): List<String> {
        val ytTable = YtTable(inputData.tablePath)

        val items = mutableSetOf<String>()
        ytProvider.getOperator(inputData.ytCluster).readTable(
            ytTable,
            YtTableRow(listOf(BUNDLE_ID, DOWNLOAD_DEEPLINK, REGION_NAME))
        ) { row: YtTableRow ->
            if (row.valueOf(DOWNLOAD_DEEPLINK) != "" && row.valueOf(REGION_NAME) == "ru") {
                items.add(row.valueOf(BUNDLE_ID))
            }
        }
        return items.toList()
    }

}
