package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.container.AdsSelectionCriteria
import ru.yandex.direct.core.entity.banner.model.DynamicBanner
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

data class DisablingParams(
    val cids: List<Long>,
    val operatorUid: Long?
)

/**
 * Ваншот для отключения лишних ДО баннеров.
 *
 * Ищет в товарных кампаниях одинаковые ДО и отключает все кроме самого старого в каждой группе.
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "xy6er", "artem-ustinov")
class SuspendExtraEcomDynamicBannersOneshot @Autowired constructor(
    private val campaignsRepository: CampaignTypedRepository,
    private val bannerService: BannerService,
    private val uacBannerService: UacBannerService
): ShardedOneshot<DisablingParams, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(SuspendExtraEcomDynamicBannersOneshot::class.java)
    }

    override fun validate(inputData: DisablingParams) = validateObject(inputData) {
        listProperty(inputData::cids)
            .check(CollectionConstraints.notEmptyCollection())
            .checkEach(CommonConstraints.validId())
        property(inputData::operatorUid) {
            check(CommonConstraints.validId(), When.notNull())
        }
    }

    override fun execute(inputData: DisablingParams, prevState: Void?, shard: Int): Void? {
        logger.info("Running on shard $shard")
        val campaigns = campaignsRepository.getTypedCampaigns(shard, inputData.cids)
            .filterIsInstance<DynamicCampaign>()
            .filter { it.metatype == CampaignMetatype.ECOM }
        campaigns.forEach { handleSingleCampaign(it, inputData.operatorUid) }
        return null
    }

    private fun handleSingleCampaign(campaign: DynamicCampaign, operatorUid: Long?) {
        logger.info("Handling campaign ${campaign.id}")
        val operator = operatorUid ?: campaign.agencyUid ?: campaign.uid
        val clientId = ClientId.fromLong(campaign.clientId)

        if (operator == null || clientId == null) {
            logger.error("Can't get operator or client for campaign ${campaign.id}")
        }

        val selectionCriteria = AdsSelectionCriteria()
            .withCampaignIds(campaign.id)
            .withTypes(BannersBannerType.dynamic)
        val banners = bannerService.getBannersBySelectionCriteria(operator, clientId, selectionCriteria, LimitOffset.maxLimited())
            .filterIsInstance<DynamicBanner>()
            .filter { hasStateOfInterest(it) }

        logger.info("Found ${banners.size} banners for campaign ${campaign.id}")

        banners
            .groupBy { it.body }
            .forEach{ (body, bannersGroup) ->
                if (bannersGroup.size <= 1) {
                    return
                }
                logger.info("Handle group with body \"$body\" having size ${bannersGroup.size}")
                handleBannersGroup(clientId, operator, bannersGroup)
            }
    }

    private fun handleBannersGroup(clientId: ClientId, operatorUid: Long, banners: List<DynamicBanner>) {
        val bannerIdsToSuspend = banners
            .map { it.id }
            .sortedBy { it }
            .toMutableList()
        logger.info("Banner IDs to suspend: $bannerIdsToSuspend")
        bannerIdsToSuspend.removeFirst() // Выкидываем первого, он остаётся активным
        if (bannerIdsToSuspend.isNotEmpty()) {
            uacBannerService.suspendResumeBanners(clientId, operatorUid, false, bannerIdsToSuspend)
        }
    }

    private fun hasStateOfInterest(banner: DynamicBanner) =
        (banner.statusShow == true)
            && (banner.statusActive == true)
            && (banner.statusArchived == false)
}
