package ru.yandex.direct.oneshot.oneshots.uc

import java.util.concurrent.TimeUnit
import org.springframework.stereotype.Component
import ru.yandex.bolts.collection.impl.EmptyMap
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.MigrationResult
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.YdbGrutMigrationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeBooleanNodeImpl
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest
import ru.yandex.yt.ytclient.proxy.request.CreateNode
import ru.yandex.yt.ytclient.proxy.request.ObjectType
import ru.yandex.yt.ytclient.tables.ColumnSchema
import ru.yandex.yt.ytclient.tables.ColumnSortOrder
import ru.yandex.yt.ytclient.tables.ColumnValueType
import ru.yandex.yt.ytclient.tables.TableSchema


abstract class BaseUacConverterYtRepository<T>(
    private val ytProvider: YtProvider,
) {
    private val resultTableCluster = YtCluster.HAHN
    abstract val resultTableColumns: List<ColumnSchema>
    fun checkIfInputTableExists(ytCluster: YtCluster, tablePath: String): Boolean {
        return ytProvider.getOperator(ytCluster).exists(YtTable(tablePath))
    }

    fun createResultTableIfNotExists(resultTablePath: String) {
        val client = ytProvider.getDynamicOperator(resultTableCluster).ytClient
        if (client.existsNode(resultTablePath).get(30, TimeUnit.SECONDS)) return

        val columnSchemaBuilder = YTree.listBuilder()
        resultTableColumns
            .forEach {
                val name = it.name
                val type = it.type
                val required = false
                val sortOrder = it.sortOrder

                columnSchemaBuilder
                    .beginMap()
                    .key("name").value(name)
                    .key("type").value(type.getName())
                    .key("required").value(required)
                if (sortOrder != null) {
                    columnSchemaBuilder.key("sort_order").value(sortOrder.getName())
                }
                columnSchemaBuilder.endMap()
            }
        val columnSchema = columnSchemaBuilder.buildList()
        val createSchema = YTree.listBuilder().buildList()
        createSchema.addAll(columnSchema.asList())

        createSchema.putAttribute("unique_keys", YTreeBooleanNodeImpl(true, EmptyMap()))
        createSchema.putAttribute("strict", YTreeBooleanNodeImpl(true, EmptyMap()))
        val tableAttributes = YTree.mapBuilder()
            .key("dynamic").value(true)
            .key("schema").value(createSchema)
            .buildMap()
            .asMap()

        val createNode = CreateNode(resultTablePath, ObjectType.Table, tableAttributes).apply { isRecursive = true }
        client.createNode(createNode).get(30, TimeUnit.SECONDS)
        client.mountTable(resultTablePath).get(30, TimeUnit.SECONDS)
    }

    open fun writeResults(resultTablePath: String, migrationResults: List<T>, columns: List<ColumnSchema> = resultTableColumns) {
        if (migrationResults.isEmpty()) return
        val operator = ytProvider.getDynamicOperator(resultTableCluster)
        val resultTableSchema = TableSchema.builder().addAll(columns).build()
        operator.runInTransaction {
            val upsertRequest = ModifyRowsRequest(resultTablePath, resultTableSchema)
            migrationResults.forEach { migrationResult ->
                upsertRequest
                    .addInsert(migrationResultToTableRow(migrationResult))
            }
            it.modifyRows(upsertRequest)
        }
    }

    abstract fun migrationResultToTableRow(migrationResult: T): List<Any?>
    fun getCampaignIdsFromYtTable(ytCluster: YtCluster, tablePath: String, startRow: Long, lastRow: Long): List<Long> {
        val campaignIds = mutableListOf<Long>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(YtTable(tablePath), { campaignIds.add(it.cid!!) }, InputTableRow(), startRow, lastRow)
        return campaignIds
    }

}

class InputTableRow : YtTableRow(listOf(CID)) {
    companion object {
        private val CID = YtField("cid", Long::class.java)
    }

    val cid: Long?
        get() = valueOf(CID)
}

/**
 * Yt-репозиторий для чтения id кампаний из входной таблицы и для записи результата миграции в выходную таблицу
 */
@Component
class UacConverterYtRepository(
    ytProvider: YtProvider,
) : BaseUacConverterYtRepository<MigrationResult>(ytProvider) {
    override val resultTableColumns = listOf(
        ColumnSchema("cid", ColumnValueType.UINT64, ColumnSortOrder.ASCENDING),
        ColumnSchema("status", ColumnValueType.STRING),
        ColumnSchema("message", ColumnValueType.STRING))


    override fun migrationResultToTableRow(migrationResult: MigrationResult) =
        listOf(migrationResult.cid, if (migrationResult.isOk) "OK" else "ERROR", migrationResult.message)
}


/**
 * Yt-репозиторий для чтения id кампаний из входной таблицы и для записи результата миграции в выходную таблицу
 */
@Component
class YdbGrutConverterYtRepository(
    ytProvider: YtProvider,
) : BaseUacConverterYtRepository<YdbGrutMigrationResult>(ytProvider) {
    override val resultTableColumns = listOf(
        ColumnSchema("cid", ColumnValueType.UINT64, ColumnSortOrder.ASCENDING),
        ColumnSchema("ydbCampaignId", ColumnValueType.STRING),
        ColumnSchema("ydbDirectCampaignStatus", ColumnValueType.INT64),
        ColumnSchema("status", ColumnValueType.STRING),
        ColumnSchema("message", ColumnValueType.STRING))

    private val resultTableColumnsWithoutYdbId = listOf(
        ColumnSchema("cid", ColumnValueType.UINT64, ColumnSortOrder.ASCENDING),
        ColumnSchema("status", ColumnValueType.STRING),
        ColumnSchema("message", ColumnValueType.STRING))


    override fun writeResults(resultTablePath: String, migrationResults: List<YdbGrutMigrationResult>, columns: List<ColumnSchema>) {
        val (campaignsWithYdbId, campaignsWithoutYdbId) = migrationResults.partition { it.ydbCampaignId != null }

        super.writeResults(resultTablePath, campaignsWithYdbId, resultTableColumns)
        // если для кампании второй раз будет запущена миграция, то ydbCampaignId не будет в результатах,и он перезатрется
        // чтобы не перезатиралось, строки без ydbId будут записываться без этой колонки
        super.writeResults(resultTablePath, campaignsWithoutYdbId, resultTableColumnsWithoutYdbId)
    }

    override fun migrationResultToTableRow(migrationResult: YdbGrutMigrationResult) =
        if (migrationResult.ydbCampaignId != null) {
            listOf(migrationResult.cid, migrationResult.ydbCampaignId, migrationResult.ydbDirectCampaignStatus, migrationResult.status.name, migrationResult.message)
        } else {
            listOf(migrationResult.cid, migrationResult.status.name, migrationResult.message)
        }

}
