package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEStrategyName
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacYdbCampaign
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.TargetType
import ru.yandex.direct.core.entity.uac.model.UacStrategy
import ru.yandex.direct.core.entity.uac.model.UacStrategyData
import ru.yandex.direct.core.entity.uac.model.UacStrategyName
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.client.Schema


data class UacFillStrategyParam(
    val ytCluster: YtCluster,
    val tablePath: String,
)

data class UacFillStrategyState(
    val lastRow: Long
)

/**
 * Ваншот для заполнения поля strategy у старых кампаний
 * После https://st.yandex-team.ru/DIRECT-159516 РМП-кампании перешли на поле strategy для создания стратегий.
 * Однако старые кампании все еще живут без этого поля и для определения правильной стратегии приходится использовать костыли.
 * Помимо этого фронту приходится мириться с двумя разными способами описания стратегии, что не очень хорошо.
 * В этом ваншоте происходит миграция старых данных на новый лад.
 */
@Component
@Retries(5)
@Multilaunch
class UacFillStrategyOneshot(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val shardHelper: ShardHelper,
    private val featureService: FeatureService,
    private val grutApiService: GrutApiService,
    private val grutTransactionProvider: GrutTransactionProvider,
) : SimpleOneshot<UacFillStrategyParam, UacFillStrategyState?> {
    companion object {
        private val logger = LoggerFactory.getLogger(UacFillStrategyOneshot::class.java)
    }

    override fun validate(inputData: UacFillStrategyParam) = validateObject(inputData) {
        property(inputData::tablePath) {
            check(CommonConstraints.notNull())
            check(
                Constraint.fromPredicate(
                    { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                    CommonDefects.objectNotFound()
                )
            )
        }
    }

    override fun execute(inputData: UacFillStrategyParam, prevState: UacFillStrategyState?): UacFillStrategyState? {
        logger.info("Start from state=$prevState")
        val chunkSize = 1000
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(
            inputData.ytCluster, inputData.tablePath, startRow, lastRow
        )
        if (cidsChunk.isEmpty()) return null

        shardHelper.getClientIdsByCampaignIds(cidsChunk)
            .forEach { (campaignId, clientId) ->
                grutTransactionProvider.runInRetryableTransaction(3) {
                    updateStrategyInGrut(campaignId, clientId)
                }
            }
        return UacFillStrategyState(lastRow)
    }

    private fun updateStrategyInGrut(campaignId: Long, clientId: Long) {
        val grutCampaign = grutUacCampaignService.getCampaignProtoByDirectCampaignId(campaignId) ?: return
        val campaign = grutCampaign.toUacYdbCampaign()
        if (campaign.strategy != null || campaign.advType != AdvType.MOBILE_CONTENT) {
            return
        }
        val enabledFeaturesForClient = featureService.getEnabledForClientId(ClientId.fromLong(clientId))
        val uacStrategy = toUacStrategy(campaign, enabledFeaturesForClient)

        val briefUpdated = Schema.TCampaign.newBuilder()
            .apply {
                meta = Schema.TCampaignMeta.newBuilder().setId(grutCampaign.meta.id).build()
                spec = Campaign.TCampaignSpec.newBuilder(grutCampaign.spec)
                    .apply {
                        campaignBrief = Campaign.TCampaignBrief.newBuilder(grutCampaign.spec.campaignBrief)
                            .apply {
                                strategy = Campaign.TBriefStrategy.newBuilder(grutCampaign.spec.campaignBrief.strategy)
                                    .apply {
                                        strategyName = uacStrategy.uacStrategyName.toEStrategyName()
                                        strategyData =
                                            UacGrutCampaignConverter.buildStrategyData(uacStrategy.uacStrategyData)
                                    }.build()
                            }.build()
                    }.build()
            }.build()

        grutApiService.briefGrutApi.updateBrief(
            briefUpdated,
            setPaths = listOf(
                "/spec/campaign_brief/strategy/strategy_name", "/spec/campaign_brief/strategy/strategy_data"
            )
        )
    }

    private fun toUacStrategy(campaign: UacYdbCampaign, enabledFeaturesForClient: Set<String>): UacStrategy {
        val strategyName = if (campaign.targetId == TargetType.CPC || campaign.skadNetworkEnabled == true) {
            UacStrategyName.AUTOBUDGET_AVG_CLICK
        } else if (enabledFeaturesForClient.contains(FeatureName.UAC_FIX_CPA_STRATEGY_ENABLED.getName())) {
            UacStrategyName.AUTOBUDGET_AVG_CPA
        } else {
            UacStrategyName.AUTOBUDGET_AVG_CPI
        }
        return UacStrategy(
            uacStrategyName = strategyName,
            uacStrategyData = UacStrategyData(
                payForConversion = campaign.targetId == TargetType.INSTALL,
                avgCpa = if (strategyName == UacStrategyName.AUTOBUDGET_AVG_CPA) campaign.cpa else null,
                avgCpi = if (strategyName == UacStrategyName.AUTOBUDGET_AVG_CPI) campaign.cpa else null,
                avgBid = if (strategyName == UacStrategyName.AUTOBUDGET_AVG_CLICK) campaign.cpa else null,
            )
        )
    }
}
