package ru.yandex.direct.oneshot.oneshots.uc

import com.fasterxml.jackson.core.type.TypeReference
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEGenders
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toGenders
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.Gender
import ru.yandex.direct.core.grut.api.BriefGrutApi
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.client.Schema

data class UacFixGenderParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    val checkExpectedGenders: Boolean,
)

data class UacFixGenderState(
    val lastRow: Long
)

data class UacFixGenderObject(
    val campaignId: Long,
    val expectedGenders: String,
    val newGenders: String,
)

class UacFixGenderRow : YtTableRow(listOf(CID)) {
    companion object {
        private val CID = YtField("cid", Long::class.java)
        private val EXPECTED_GENDERS = YtField("expected_genders", String::class.java)
        private val NEW_GENDERS = YtField("new_genders", String::class.java)
    }

    val cid: Long?
        get() = valueOf(CID)
    val expectedGenders: String?
        get() = valueOf(EXPECTED_GENDERS)
    val newGenders: String?
        get() = valueOf(NEW_GENDERS)
}

@Component
@Approvers("mspirit", "elwood", "khuzinazat", "kozobrodov")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class UacFixGenderOneshot(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val ytProvider: YtProvider,
    grutContext: GrutContext,
) : SimpleOneshot<UacFixGenderParam, UacFixGenderState?> {

    companion object {
        private val logger = LoggerFactory.getLogger(UacFixGenderOneshot::class.java)
    }

    private val briefGrutApi = BriefGrutApi(grutContext)

    override fun validate(inputData: UacFixGenderParam) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: UacFixGenderParam, prevState: UacFixGenderState?): UacFixGenderState? {
        logger.info("Start from state=$prevState")
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + 1
        val objects = getRowsFromYtTable(
            inputData.ytCluster,
            inputData.tablePath,
            startRow,
            lastRow
        )
        if (objects.isEmpty()) {
            return null
        }
        val obj = objects[0]
        val cid = obj.campaignId
        logger.info("Processing campaign: $cid")
        try {
            grutTransactionProvider.runInRetryableTransaction(3) {
                val brief = briefGrutApi.getBrief(cid, listOf("/spec/campaign_brief/socdem/genders"))
                if (brief == null) {
                    logger.info("Campaign $cid not found")
                    return@runInRetryableTransaction
                }
                val currentGenders = brief.spec.campaignBrief.socdem.gendersList.toGenders().toSet()
                if (inputData.checkExpectedGenders) {
                    val expectedGenders = JsonUtils.fromJson(obj.expectedGenders,
                        object : TypeReference<List<Gender>>() {}).toSet()
                    if (currentGenders != expectedGenders) {
                        logger.info("Current genders in GrUT: $currentGenders, " +
                            "but expected $expectedGenders, skipping campaign")
                        return@runInRetryableTransaction
                    }
                }
                val newGenders = JsonUtils.fromJson(obj.newGenders,
                    object : TypeReference<List<Gender>>() {})
                if (currentGenders == newGenders.toSet()) {
                    logger.info("Current and new genders are equals: $currentGenders, skipping campaign")
                    return@runInRetryableTransaction
                }
                briefGrutApi.updateBrief(
                    brief = Schema.TCampaign.newBuilder().apply {
                        meta = Schema.TCampaignMeta.newBuilder().setId(cid).build()
                        spec = Campaign.TCampaignSpec.newBuilder().apply {
                            campaignBrief = Campaign.TCampaignBrief.newBuilder().apply {
                                socdem = Campaign.TCampaignBrief.TSocdem.newBuilder()
                                    .addAllGenders(newGenders.toEGenders()).build()
                                briefSynced = false
                            }.build()
                        }.build()
                    }.build(),
                    setPaths = listOf("/spec/campaign_brief/socdem/genders"))
            }
        } catch (e: Exception) {
            logger.error("Failed to handle campaign $cid, skipping", e)
        }
        return UacFixGenderState(lastRow)
    }

    private fun getRowsFromYtTable(
        ytCluster: YtCluster,
        tablePath: String,
        startRow: Long,
        lastRow: Long
    ): List<UacFixGenderObject> {
        val result = mutableListOf<UacFixGenderObject>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(
                YtTable(tablePath),
                { result.add(UacFixGenderObject(it.cid!!, it.expectedGenders!!, it.newGenders!!)) },
                UacFixGenderRow(),
                startRow,
                lastRow
            )
        return result
    }
}
