package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.service.CampaignContentUpdateService
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.Param
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.State
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.UcTgoConverterOneshot
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

/**
 * Ваншот для пересчета статусов контента в ydb
 * Нужен для случаев добавления новых типов контентов, когда расчет статусов появляется позже
 */
@Component
@Approvers("mspirit", "dimitrovsd")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class UacRecalculateCampaignContentStatusesOneshot @Autowired constructor(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val campaignContentUpdateService: CampaignContentUpdateService,
) : ShardedOneshot<Param, State?> {
    companion object {
        private val logger = LoggerFactory.getLogger(UcTgoConverterOneshot::class.java)
        private const val CHUNK_SIZE = 100
    }

    override fun validate(inputData: Param) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Param, prevState: State?, shard: Int): State? {
        logger.info("Start from state=$prevState, shard=$shard")
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + CHUNK_SIZE
        val campaignIdsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        val campaignIdsChunkShard = campaignTypedRepository.getSafely(shard, campaignIdsChunk, CommonCampaign::class.java).map { it.id }
        campaignContentUpdateService
            .updateCampaignContentsAndCampaignFlags(shard, campaignIdsChunkShard)

        if (campaignIdsChunk.size < CHUNK_SIZE) {
            return null
        }
        return State(lastRow)
    }
}
