package ru.yandex.direct.oneshot.oneshots.uc

import com.google.common.collect.Lists
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService
import ru.yandex.direct.core.entity.campaign.model.CampaignSource
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.grut.api.BriefBanner
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SafeOneshot
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.grut.objects.proto.Banner.TBannerSpec.EBannerStatus

/**
 * Ваншот для отправки UAC баннеров на модерацию
 */
@Component
@Multilaunch
@Retries(5)
@PausedStatusOnFail
@SafeOneshot
class UacSendBannersToModerationOneshot @Autowired constructor(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val shardHelper: ShardHelper,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val bannerTypedRepository: BannerTypedRepository,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    private val bannerModerateService: BannerModerateService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService,
) : SimpleOneshot<Input, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(UacSendBannersToModerationOneshot::class.java)
    }

    override fun validate(inputData: Input) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Input, prevState: State?): State? {
        logger.info("Start from state=$prevState")
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + 1
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(
            inputData.ytCluster,
            inputData.tablePath,
            startRow,
            lastRow
        )
        if (cidsChunk.isEmpty()) {
            return null
        }
        val campaignId = cidsChunk.first()
        logger.info("Processing campaign: $campaignId")

        val shard = shardHelper.getShardByCampaignId(campaignId)
        if (shard == ShardSupport.NO_SHARD) {
            logger.error("Campaign with id $campaignId not found")
            return State(lastRow)
        }
        val campaigns = campaignTypedRepository.getSafely(shard, listOf(campaignId), CommonCampaign::class.java)
        if (campaigns.isEmpty()) {
            logger.error("Campaign with id $campaignId not found")
            return State(lastRow)
        }
        val campaign = campaigns.first()

        if (campaign.source != CampaignSource.UAC) {
            logger.info("Campaign $campaignId is not UC")
            return State(lastRow)
        }

        val clientId = ClientId.fromLong(campaign.clientId)
        val operatorUid = inputData.operator ?: campaign.agencyUid ?: campaign.uid

        val bidsToModerate = try {
            findBannersToSendToModeration(shard, clientId, campaignId)
        } catch (e: Throwable) {
            logger.error("Processing campaign $campaignId failed", e)
            return State(lastRow)
        }

        if (bidsToModerate.isEmpty()) {
            logger.info("No bids to moderate")
            return State(lastRow)
        }
        logger.info("Campaign $campaignId needs processing")
        logger.info("Bids to process: $bidsToModerate")

        val dryRun = inputData.dryRun ?: true
        if (!dryRun) {
            val successful = sendToModeration(operatorUid, clientId, bidsToModerate.toList(), campaignId)
            logger.info("Moderation in campaign $campaignId is successful: $successful")
        } else {
            logger.info("Run is dry, no processing")
        }

        return State(lastRow)
    }

    private fun findBannersToSendToModeration(shard: Int, clientId: ClientId, campaignId: Long): Set<Long> {
        val uacCampaignService = uacCampaignServiceHolder.getUacCampaignService(true)
        val campaign = uacCampaignService
            .getCampaignById(campaignId.toIdString())

        if (campaign == null) {
            logger.error("Campaign $campaignId not found in GRuT")
            return emptySet()
        }
        val campaignStatuses = uacCampaignService.getCampaignStatuses(clientId, campaignId, campaign)
        if (campaignStatuses == null) {
            logger.error("Campaign $campaignId statuses not found")
            return emptySet()
        }
        if (campaignStatuses.status == Status.DRAFT) {
            logger.error("Campaign $campaignId is draft")
            return emptySet()
        }

        val bidsToModerate = bannerTypedRepository.getBannersByCampaignIds(shard, listOf(campaignId)).asSequence()
            .filter { (it as BannerWithSystemFields).statusArchived == false }
            .filter { (it as BannerWithSystemFields).statusModerate == BannerStatusModerate.NEW }
            .map { it.id }
            .toSet()

        logger.info("Found ${bidsToModerate.size} unmoderated banners")

        return bidsToModerate
    }

    private fun sendToModeration(
        operatorUid: Long,
        clientId: ClientId,
        bannerIds: List<Long>,
        campaignId: Long,
    ): Boolean {
        var isModerationSuccessful = true
        for (chunk in Lists.partition(bannerIds, 100)) {
            val result = bannerModerateService.moderateBanners(clientId, operatorUid, chunk)
            if (result == null || result.validationResult.hasAnyErrors())
                logger.error(
                    "Can't send to moderation banner ids: {}, first error is: {}",
                    chunk, result.validationResult.flattenErrors()[0].toString()
                )
            isModerationSuccessful = false
        }

        if (isModerationSuccessful) {
            try {
                updateStatusByDirectAdIds(bannerIds)
            } catch (e: Throwable) {
                logger.error("Can't update banners in campaign $campaignId", e)
            }
        }

        return isModerationSuccessful
    }

    private fun updateStatusByDirectAdIds(bannerIds: Collection<Long>) {
        // todo #частичный_update
        grutTransactionProvider.runInRetryableTransaction(3) {
            val bannerById = grutApiService.briefBannerGrutApi
                .getBanners(bannerIds)
                .associateBy { it.meta.id }
            bannerIds.map {
                val oldBanner = bannerById[it]!!
                BriefBanner(
                    id = it,
                    adGroupId = oldBanner.meta.adGroupId,
                    briefId = oldBanner.meta.campaignId,
                    source = oldBanner.meta.source,
                    assetIds = oldBanner.spec.assetIdsList,
                    assetLinksIds = oldBanner.spec.assetLinkIdsList,
                    status = EBannerStatus.BSS_MODERATING
                )
            }
        }
    }
}
