package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.converter.getProtoEnumValueName
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SafeOneshot
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.grut.objects.proto.Banner.TBannerSpec.EBannerStatus.BSS_DELETED

data class Input(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operator: Long?,
    val dryRun: Boolean?,
)

/**
 * Ваншот для синхронизации баннеров в груте и mysql. Полезен в случае, если данные в базах разъехались относительно
 * друг друга
 */
@Component
@Approvers("mspirit", "dimitrovsd", "khuzinazat")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
@SafeOneshot
class UacSyncBannersOneshot @Autowired constructor(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val uacBannerService: UacBannerService,
    private val shardHelper: ShardHelper,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val bannerTypedRepository: BannerTypedRepository,
    private val grutApiService: GrutApiService,
) : SimpleOneshot<Input, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(this::class.java)
    }

    override fun validate(inputData: Input) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Input, prevState: State?): State? {
        logger.info("Start from state=$prevState")
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + 1
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(
            inputData.ytCluster,
            inputData.tablePath,
            startRow,
            lastRow
        )
        if (cidsChunk.isEmpty()) {
            return null
        }
        val campaignId = cidsChunk.first()
        logger.info("Processing campaign: $campaignId")

        val shard = shardHelper.getShardByCampaignId(campaignId)
        if (shard == NO_SHARD) {
            logger.error("Campaign with id $campaignId not found")
            return State(lastRow)
        }
        val campaigns = campaignTypedRepository.getSafely(shard, listOf(campaignId), CommonCampaign::class.java)
        if (campaigns.isEmpty()) {
            logger.error("Campaign with id $campaignId not found")
            return State(lastRow)
        }
        val campaign = campaigns.first()

        if (!AvailableCampaignSources.isUC(campaign.source)) {
            logger.info("Campaign $campaignId is not UC")
            return State(lastRow)
        }

        val clientId = ClientId.fromLong(campaign.clientId)
        val operatorUid = inputData.operator ?: campaign.agencyUid ?: campaign.uid

        val bidsToArchive = try {
            findAbsentBanners(shard, campaignId)
        } catch (e: Throwable) {
            logger.error("Processing campaign $campaignId failed", e)
            return State(lastRow)
        }

        if (bidsToArchive.isEmpty()) {
            logger.info("No bids to archive")
            return State(lastRow)
        }
        logger.info("Campaign $campaignId needs processing")
        logger.info("Bids to process: $bidsToArchive")

        val dryRun = inputData.dryRun ?: true
        if (!dryRun) {
            processBanners(bidsToArchive, clientId, operatorUid)
        } else {
            logger.info("Run is dry, no processing")
        }

        return State(lastRow)
    }

    private fun findAbsentBanners(shard: Int, campaignId: Long): Set<Long> {
        val bidsInMysql = bannerTypedRepository.getBannersByCampaignIds(shard, listOf(campaignId)).asSequence()
            .filter { (it as BannerWithSystemFields).statusArchived == false }
            .map { it.id }
            .toSet()
        val bidsInGrut = grutApiService.briefBannerGrutApi.selectBanners(
            filter = "[/meta/campaign_id] = $campaignId " +
                "AND [/spec/status] != \"${getProtoEnumValueName(BSS_DELETED)}\"",
            index = "banners_by_campaign",
            attributeSelector = listOf("/meta/id")
        ).asSequence().map { it.meta.id }.toSet()


        logger.info("Found ${bidsInMysql.size} unarchived banners in MySQL and ${bidsInGrut.size} banners in GrUT")

        return bidsInMysql - bidsInGrut
    }

    private fun processBanners(bannerIdsToRemove: Set<Long>, clientId: ClientId, operatorUid: Long) {
        // Останавливаем баннеры
        val suspendResult = uacBannerService.suspendResumeBanners(clientId, operatorUid, false, bannerIdsToRemove)
        suspendResult.logErrors("banners suspension")

        val (deleteMassResult, archiveMassResult) = uacBannerService
            .deleteAndArchiveBanners(operatorUid, clientId, bannerIdsToRemove)
        deleteMassResult.logErrors("banners deletion")
        archiveMassResult.logErrors("banners archivation")

        val successfulResultBannerIds = mutableSetOf<Long>()
        successfulResultBannerIds.addAll(getSuccessResults(deleteMassResult) + getSuccessResults(archiveMassResult))

        val failedResultBannerIds = bannerIdsToRemove - successfulResultBannerIds
        if (failedResultBannerIds.isNotEmpty()) {
            logger.error("Failed to remove banners: $failedResultBannerIds")
        }
    }

    private fun getSuccessResults(massResult: MassResult<Long>?): Set<Long> {
        return if (massResult?.result == null) {
            emptySet()
        } else massResult.result
            .filterNotNull()
            .filter { it.isSuccessful }
            .mapNotNull { it.result }
            .toSet()
    }

    private fun MassResult<Long>.logErrors(action: String = "action") {
        if (!this.isSuccessful) {
            logger.error("Result of $action was unsuccessful: {0}", this.validationResult.flattenErrors())
        }
    }
}
