package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.model.YtCluster

data class UacSyncPhrasesParam(
    val ytCluster: YtCluster,
    val tablePath: String,
)

data class UacSyncPhrasesState(
    val lastRow: Long
)

/**
 * Ваншот для синхронизации фраз uc/uac-кампаний из mysql в ydb
 */
@Component
@Multilaunch
@Approvers("bratgrim", "khuzinazat")
class UacSyncPhrasesOneshot @Autowired constructor(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val keywordRepository: KeywordRepository,
    private val adGroupRepository: AdGroupRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
) : ShardedOneshot<UacSyncPhrasesParam, UacSyncPhrasesState?> {
    companion object {
        private val logger = LoggerFactory.getLogger(UacSyncPhrasesOneshot::class.java)
        private const val CHUNK_SIZE = 1000L
    }

    override fun validate(inputData: UacSyncPhrasesParam) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: UacSyncPhrasesParam, prevState: UacSyncPhrasesState?, shard: Int): UacSyncPhrasesState? {
        logger.info("Start from state=$prevState, shard=$shard")
        val chunkSize = CHUNK_SIZE
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        if (cidsChunk.isEmpty()) return null
        val cidsChunkShard = campaignTypedRepository.getSafely(shard, cidsChunk.distinct(), CommonCampaign::class.java).map { it.id }
        if (cidsChunkShard.isEmpty()) return null

        val ydbCampaignIdByCid = uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(cidsChunkShard)
        val adGroupIdByCid = adGroupRepository.getAdGroupIdsByCampaignIds(shard, cidsChunkShard)
        cidsChunkShard.forEach { cid ->
            val ydbCampaignId = ydbCampaignIdByCid[cid]
            if (ydbCampaignId == null) {
                logger.error("Campaign $cid in shard $shard has no ydb campaign")
                return@forEach
            }
            val adGroupId = adGroupIdByCid[cid]?.maxOrNull()
            if (adGroupId == null) {
                logger.warn("Campaign $cid in shard $shard has no ad groups")
                return@forEach
            }
            val ydbCampaign = uacYdbCampaignRepository.getCampaign(ydbCampaignId)
            if (ydbCampaign?.briefSynced != true) {
                logger.info("Campaign $cid in shard $shard is not synced")
                return@forEach
            }
            val keywords = keywordRepository.getKeywordsByAdGroupId(shard, adGroupId).map { it.phrase }
            if (keywords.isNullOrEmpty()) {
                logger.error("Campaign $cid in shard $shard has no keywords")
                return@forEach
            }
            val modelChanges = KtModelChanges<String, UacYdbCampaign>(ydbCampaignId)
            modelChanges.process(UacYdbCampaign::keywords, keywords)
            uacYdbCampaignRepository.update(modelChanges)
        }
        return if (cidsChunk.size < chunkSize) null else UacSyncPhrasesState(lastRow)
    }
}
