package ru.yandex.direct.oneshot.oneshots.uc

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames.UC_UPDATE_ADS_CHUNK_SIZE
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.model.YtCluster

data class Param(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operator: Long?,
)

data class State(
    val lastRow: Long
)


/**
 * Ваншот для создания обновления баннеров для uac-кампаний
 * Кладет кампании для обновления в dbQueue
 * Само обновление будет в джобе UpdateAdsJob
 *
 * параметр operator должен быть таким, чтобы у него были права на обновление кампании. например, супер пользователь Директа
 */
@Component
@Approvers("mspirit", "khuzinazat", "bratgrim", "pavelkataykin", "andreypav", "babashev", "kozobrodov")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class UacUpdateAdsOneshot(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val uacBannerService: UacBannerService,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val shardHelper: ShardHelper,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    private val uacDbDefineService: UacDbDefineService,
    private val campaignTypedRepository: CampaignTypedRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : SimpleOneshot<Param, State?> {
    private val chunkSizeProperty: PpcProperty<Int> = ppcPropertiesSupport.get(UC_UPDATE_ADS_CHUNK_SIZE)

    companion object {
        private val logger = LoggerFactory.getLogger(UacUpdateAdsOneshot::class.java)
    }

    override fun validate(inputData: Param) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Param, prevState: State?): State? {
        logger.info("Start from state=$prevState")
        val chunkSize = chunkSizeProperty.getOrDefault(100)
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(
            inputData.ytCluster, inputData.tablePath, startRow, lastRow
        )
        if (cidsChunk.isEmpty()) return null

        val campaignIdsYdb = uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(cidsChunk.distinct())

        val campaignByIds = mutableMapOf<Long, CommonCampaign>()
        shardHelper.groupByShard(cidsChunk.distinct(), ShardKey.CID).forEach { shard, cids ->
            campaignByIds.putAll(campaignTypedRepository.getIdToModelSafely(shard, cids, CommonCampaign::class.java))
        }

        shardHelper.getClientIdsByCampaignIds(cidsChunk.distinct())
            .forEach { (campaignId, clientId) ->
                val uacCampaignId = if (campaignIdsYdb.containsKey(campaignId)) {
                    logger.info("Campaign $campaignId in YDB")
                    campaignIdsYdb[campaignId]!!
                } else {
                    logger.info("Campaign $campaignId in GrUT")
                    campaignId.toString()
                }
                val useGrut = uacDbDefineService.useGrut(uacCampaignId)
                uacCampaignServiceHolder.getUacBannerService(useGrut).updateBriefSynced(uacCampaignId, false)
                if (!campaignByIds.containsKey(campaignId)) {
                    logger.error("Campaign $campaignId not found in MySQL!")
                    return@forEach
                }
                val campaign = campaignByIds[campaignId]!!
                val operatorUid = inputData.operator ?: campaign.agencyUid ?: campaign.uid
                uacBannerService.updateAdsDeferred(ClientId.fromLong(clientId), operatorUid, uacCampaignId)
            }
        return if (cidsChunk.size < chunkSize) null else State(lastRow)
    }
}
