package ru.yandex.direct.oneshot.oneshots.uc

import java.time.Duration
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.UC_UPDATE_CAMPAIGN_TARGET_STATUS_IDLE_TIME
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief.EBriefStatus
import ru.yandex.grut.objects.proto.client.Schema

data class UacCampaignsParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    val lastRow: Long = 0L,
)

data class UacCampaignsState(
    var lastRow: Long = 0L,
    var deletedCount: Int = 0
)

/**
 * Ваншот для обновления target_status у грутовых заявок
 * Получает с подготовленный в Yt таблице список cid и по нему отбирает кампании в mysql и заявки в grut,
 * Затем смеряет их statusShow и target_status:
 * - если статусы отличаются и кампания не draft - меняет в груте статус на соответствующий из mysql
 * - если статусы одинаковые, кампания draft и статус = started - меняет в груте статус на остановленный
 */
@Component
@Approvers("pavelkataykin", "khuzinazat", "bratgrim", "mspirit", "a-dubov", "ppalex", "gerdler", "maxlog", "kuhtich")
@Retries(5)
@Multilaunch
@PausedStatusOnFail
class UacUpdateCampaignTargetStatusOneshot(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val grutApiService: GrutApiService,
    private val grutTransactionProvider: GrutTransactionProvider,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : SimpleOneshot<UacCampaignsParam, UacCampaignsState> {
    companion object {
        private val CID = YtField("cid", Long::class.javaObjectType)
        private const val CHUNK_LIMIT: Int = 1
    }

    private val logger = LoggerFactory.getLogger(this.javaClass)
    private val idleTimeProperty =
        ppcPropertiesSupport.get(UC_UPDATE_CAMPAIGN_TARGET_STATUS_IDLE_TIME, Duration.ofMinutes(1))

    override fun validate(
        inputData: UacCampaignsParam
    ): ValidationResult<UacCampaignsParam, Defect<*>> {
        val validateBuilder = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(
            ytProvider,
            validateBuilder,
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(
        inputData: UacCampaignsParam,
        prevState: UacCampaignsState?
    ): UacCampaignsState? {
        val state = prevState ?: UacCampaignsState(inputData.lastRow)

        val chunkSize = maxOf(1, CHUNK_LIMIT)
        val startRow = state.lastRow
        val finishRow = chunkSize + startRow

        val cidsChunk = getCampaignIdsByShardFromYt(inputData, startRow, finishRow)
        if (cidsChunk.isEmpty()) return null

        val campaignIdsByShard = shardHelper.groupByShard(cidsChunk, ShardKey.CID).shardedDataMap
        campaignIdsByShard.forEach { (shard, campaignIds) ->
            campaignIds.forEach { campaignId ->
                fixCampaignStatus(shard, campaignId)

                Thread.sleep(maxOf(10L, idleTimeProperty.getOrDefault(500L)))
            }
        }

        return UacCampaignsState(finishRow)
    }

    /**
     * Берет кампанию из mysql и заявку из grut, сравнивает их статусы (statusShow и target_status) и если
     * они расходятся - обновляет статус в grut заявке
     */
    private fun fixCampaignStatus(
        shard: Int,
        campaignId: Long,
    ) {
        val statusShow = getCampaignStatusShow(shard, campaignId)
        if (statusShow == null) {
            logger.warn("Campaign $campaignId deleted or not uac")
            return
        }

        grutTransactionProvider.runInRetryableTransaction(3) {
            val grutCampaign = getGrutCampaign(campaignId)
            if (grutCampaign == null || !grutCampaign.spec.campaignBrief.hasTargetStatus()) {
                logger.warn("Campaign $campaignId does not exist or does not have target_status in grut")
                return@runInRetryableTransaction
            }
            val grutStatusShow = grutCampaign.spec.campaignBrief.targetStatus
            val campaignBriefIsDraft = grutCampaign.spec.status == Campaign.TCampaignSpec.ECampaignStatus.CS_DRAFT

            val correctTargetStatus =
                if (statusShow == CampaignsStatusshow.Yes && !campaignBriefIsDraft) EBriefStatus.BS_STARTED
                // Если кампания черновик - расхождение статусов не страшно, они исправятся при запуске кампании
                else if (statusShow == CampaignsStatusshow.No && campaignBriefIsDraft) grutStatusShow
                else EBriefStatus.BS_STOPPED

            val campaignLog = if (campaignBriefIsDraft) "Draft campaign" else "Campaign"
            if (grutStatusShow == correctTargetStatus) {
                logger.info(
                    "$campaignLog $campaignId already have a valid status " +
                        "in grut ${grutStatusShow.name} and mysql ${statusShow.name}"
                )
            } else {
                logger.info("$campaignLog $campaignId had $grutStatusShow target_status in grut")
                updateGrutCampaignTargetStatus(grutCampaign, correctTargetStatus)
            }
        }
    }

    private fun getGrutCampaign(campaignId: Long): Schema.TCampaign? {
        val grutCampaigns = grutUacCampaignService.getCampaigns(listOf(campaignId.toString()))
        if (grutCampaigns.isEmpty()) {
            logger.warn("Campaign $campaignId does not exist in grut")
            return null
        }
        return grutCampaigns[0]
    }

    private fun getCampaignStatusShow(
        shard: Int,
        campaignId: Long
    ): CampaignsStatusshow? = dslContextProvider.ppc(shard)
        .select(Tables.CAMPAIGNS.CID, Tables.CAMPAIGNS.STATUS_SHOW)
        .from(Tables.CAMPAIGNS)
        .where(Tables.CAMPAIGNS.CID.eq(campaignId))
        .and(Tables.CAMPAIGNS.SOURCE.eq(CampaignsSource.uac))
        .and(Tables.CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
        .fetchOne(Tables.CAMPAIGNS.STATUS_SHOW)


    /**
     * Обновляет target_status в грутовой заявке
     */
    private fun updateGrutCampaignTargetStatus(
        grutCampaign: Schema.TCampaign,
        targetStatus: EBriefStatus,
    ) {
        val briefUpdated = Schema.TCampaign.newBuilder()
            .apply {
                meta = Schema.TCampaignMeta.newBuilder().setId(grutCampaign.meta.id).build()
                spec = Campaign.TCampaignSpec.newBuilder(grutCampaign.spec)
                    .apply {
                        campaignBrief = Campaign.TCampaignBrief.newBuilder(grutCampaign.spec.campaignBrief)
                            .setTargetStatus(targetStatus)
                            .build()
                    }.build()
            }.build()

        grutApiService.briefGrutApi.updateBrief(
            briefUpdated,
            setPaths = listOf("/spec/campaign_brief/target_status")
        )
    }

    private fun getCampaignIdsByShardFromYt(
        inputData: UacCampaignsParam,
        startRow: Long,
        finishRow: Long,
    ): List<Long> {
        val campaignIds = mutableListOf<Long>()
        ytProvider.getOperator(inputData.ytCluster).readTableByRowRange(
            YtTable(inputData.tablePath),
            { campaignIds.add(it.valueOf(CID)) },
            YtTableRow(listOf(CID)),
            startRow,
            finishRow
        )
        return campaignIds
    }
}
