package ru.yandex.direct.oneshot.oneshots.uc

import org.jooq.Configuration
import org.jooq.DSLContext
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.oneshots.uc.repository.OneshotUacCampaignRepository
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class UpdateCPCParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    var lastRow: Long = 0,
    var chunkSize: Int = 2000
)

data class UpdateCPCState(
    var lastRow: Long = 0L,
    var countUpdated: Int = 0
) {
    fun increaseCountUpdatesBy(value: Int) {
        countUpdated += value
    }

    fun withLastRow(index: Long): UpdateCPCState {
        this.lastRow = index
        return this
    }
}

/**
 * Ряд CPC кампаний были созданы с ошибкой. В стратегии этих кампаний в качестве goalId указали 0.
 * Это со стороны БК трактуется как оптимизация по всем целям и делает кампанию конверсионной.
 * Oneshot исправляет эту проблему, считывает CPC кампании из предварительно отобранного списка кампаний в YT,
 * у которых goalId is not null и удаляет goalId
 */
@Component
@Approvers("pavelkataykin", "khuzinazat", "bratgrim")
@Retries(5)
@Multilaunch
@PausedStatusOnFail
class UpdateCPCCampaignGoalsToNullOneshot(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val bsResyncService: BsResyncService,
    private val oneshotUacCampaignRepository: OneshotUacCampaignRepository,
    private val shardHelper: ShardHelper,
) : SimpleOneshot<UpdateCPCParam, UpdateCPCState> {
    companion object {
        private val CID = YtField("cid", Long::class.javaObjectType)
    }

    private val logger = LoggerFactory.getLogger(this.javaClass)

    override fun validate(inputData: UpdateCPCParam): ValidationResult<UpdateCPCParam, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(ytProvider, vb, inputData.ytCluster, inputData.tablePath)
    }

    override fun execute(inputData: UpdateCPCParam, prevState: UpdateCPCState?): UpdateCPCState? {
        val state = prevState ?: UpdateCPCState(inputData.lastRow)

        val startRow = state.lastRow
        val finishRow = inputData.chunkSize + startRow

        logger.info("CPC campaign goal update launched for campaigns $startRow to $finishRow " +
            "(chunk size is ${inputData.chunkSize})")

        val updateCPCCampaignIdsByShard = getCampaignIdsByShardFromYt(inputData, startRow, finishRow)

        if (updateCPCCampaignIdsByShard.isNotEmpty()) {
            val updatedCount = updateStrategies(updateCPCCampaignIdsByShard)
            state.increaseCountUpdatesBy(updatedCount)
        }
        return if (updateCPCCampaignIdsByShard.values.sumOf { it.size } < inputData.chunkSize) {
            logger.info("${state.countUpdated} where updated in total")
            null
        } else {
            state.withLastRow(finishRow)
        }
    }

    private fun setStrategyDataGoalIdToNullForCPCCampaign(
        dslContextProvider: DslContextProvider,
        cpcCampaignsWithGoalsByShard: Map<Int, List<Long>>
    ): List<Long> {
        val updated: MutableList<Long> = ArrayList()
        cpcCampaignsWithGoalsByShard.forEach { (shard, campaignIds) ->
            dslContextProvider.ppc(shard).transaction { config: Configuration ->
                val dslContext = config.dsl()
                getCampaignsWithCPCAndGoalId(dslContext, campaignIds).forEach { (id, strategy) ->
                    val previousState = JsonUtils.toJson(strategy.strategyData)
                    strategy.strategyData.goalId = null
                    if (oneshotUacCampaignRepository.updateCampaignStrategyData(dslContext, id, strategy)) {
                        logger.info("Shard= $shard, cid= ${id}: strategy successfully updated to:" +
                            " ${JsonUtils.toJson(strategy.strategyData)} from: $previousState")
                        updated.add(id)
                    }
                }
            }
        }
        return updated
    }

    private fun updateStrategies(cpcCampaignsWithGoalsByShard: Map<Int, List<Long>>): Int {
        val campaignsIdsToResync = setStrategyDataGoalIdToNullForCPCCampaign(
            dslContextProvider,
            cpcCampaignsWithGoalsByShard)

        logger.info("${campaignsIdsToResync.size} campaigns out " +
            "of ${cpcCampaignsWithGoalsByShard.values.sumOf { it.size }} has been updated")

        val bsResyincItems = campaignsIdsToResync
            .map { BsResyncItem(BsResyncPriority.PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE, it) }
        val addedForResync = bsResyncService.addObjectsToResync(bsResyincItems)
        logger.info("$addedForResync campaigns has been queued for bs_resync_queue table")
        return campaignsIdsToResync.size
    }

    private fun getCampaignIdsByShardFromYt(
        inputData: UpdateCPCParam,
        minRow: Long,
        maxRow: Long
    ): Map<Int, List<Long>> {
        val campaignIds = mutableListOf<Long>()
        ytProvider.getOperator(inputData.ytCluster).readTableByRowRange(
            YtTable(inputData.tablePath),
            { campaignIds.add(it.valueOf(CID)) },
            YtTableRow(listOf(CID)),
            minRow,
            maxRow
        )
        logger.info("${campaignIds.size} campaigns where obtained for check")
        return shardHelper.groupByShard(campaignIds, ShardKey.CID).shardedDataMap
    }

    private fun getCampaignsWithCPCAndGoalId(
        dslContext: DSLContext,
        updateCPCCampaignsIds: List<Long>,
    ): Map<Long, DbStrategy> = oneshotUacCampaignRepository.getRMPCampaignsByIdAndStrategyNameWithGoalsFromUac(
        dslContext,
        updateCPCCampaignsIds,
        StrategyName.AUTOBUDGET_AVG_CLICK,
        true
    ).filterValues { it.isCpc && it.strategyData.goalId != null }

}
