package ru.yandex.direct.oneshot.oneshots.uc.repository

import java.time.LocalDateTime
import org.jooq.DSLContext
import org.jooq.exception.DataAccessException
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders
import ru.yandex.direct.utils.JsonUtils

@Repository
class OneshotUacCampaignRepository(
    private val dslContextProvider: DslContextProvider,
) {

    private val logger = LoggerFactory.getLogger(this.javaClass)

    companion object {
        private val strategyMapper = JooqReaderWithSupplierBuilder.builder(::DbStrategy)
            .readProperty(DbStrategy.STRATEGY_NAME,
                ReaderBuilders.fromField(CAMPAIGNS.STRATEGY_NAME)
                    .by { StrategyName.fromSource(it) })
            .readProperty(DbStrategy.STRATEGY_DATA,
                ReaderBuilders.fromField(CAMPAIGNS.STRATEGY_DATA)
                    .by { CampaignMappings.strategyDataFromDb(it) })
            .build()
    }

    fun getUacCampaignIdsWithDefectArchivedBanners(
        shard: Int,
        clientId: Long?,
    ): List<Long> {
        var condition = CAMPAIGNS.SOURCE.eq(CampaignsSource.uac)
            .and(CAMPAIGNS.TYPE.eq(CampaignsType.text))
            .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.Yes))
            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))

        if (clientId != null) {
            condition = condition.and(CAMPAIGNS.CLIENT_ID.eq(clientId))
        }

        return dslContextProvider.ppc(shard)
            .select(CAMPAIGNS.CID)
            .from(CAMPAIGNS)
            .join(BANNERS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
            .where(condition)
            .groupBy(CAMPAIGNS.CID)
            .fetch(CAMPAIGNS.CID)
    }

    fun getUacCampaignsByClientId(
        dslContext: DSLContext,
        campaignIds: List<Long>,
    ): Map<Long, List<Long>> {
        return dslContext
            .select(CAMPAIGNS.CID, CAMPAIGNS.CLIENT_ID)
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.CID.`in`(campaignIds))
            .and(CAMPAIGNS.SOURCE.eq(CampaignsSource.uac))
            .and(CAMPAIGNS.TYPE.`in`(CampaignsType.mobile_content, CampaignsType.text, CampaignsType.cpm_banner))
            .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
            .fetch { it.get(CAMPAIGNS.CLIENT_ID) to it.get(CAMPAIGNS.CID) }
            .groupBy({ it.first }, { it.second })
    }

    fun getRMPCampaignsByIdAndStrategyNameWithGoalsFromUac(
        dslContext: DSLContext,
        campaignIds: List<Long>,
        strategyName: StrategyName,
        forUpdate: Boolean = false
    ): Map<Long, DbStrategy> {
        val select = dslContext
            .select(CAMPAIGNS.CID, CAMPAIGNS.STRATEGY_NAME, CAMPAIGNS.STRATEGY_DATA)
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.CID.`in`(campaignIds))
            .and(CAMPAIGNS.STRATEGY_NAME.eq(StrategyName.toSource(strategyName)))
            .and(CAMPAIGNS.TYPE.eq(CampaignsType.mobile_content))
            .and(CAMPAIGNS.SOURCE.eq(CampaignsSource.uac))
        val result = if (forUpdate) select.forUpdate() else select
        return result
            .fetch { it.get(CAMPAIGNS.CID) to strategyMapper.fromDb(it) }
            .toMap()
    }

    fun updateCampaignStrategyData(dslContext: DSLContext, campaignId: Long, strategy: DbStrategy): Boolean {
        try {
            return dslContext.update(CAMPAIGNS)
                .set(CAMPAIGNS.STRATEGY_DATA, JsonUtils.toJson(strategy.strategyData))
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .where(CAMPAIGNS.CID.eq(campaignId))
                .execute() > 0
        } catch (ex: DataAccessException) {
            logger.error("Fatal error! Update for $campaignId has thrown an exception", ex)
        }
        return false
    }
}
