package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.env.Environment
import ru.yandex.direct.oneshot.oneshots.uc.UacConverterYtRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.client.Schema

data class GrutSetStatusForMigrCampaignsParam(
    val ytCluster: YtCluster,
    val tablePath: String,
)

@Component
@Approvers("mspirit", "dimitrovsd", "elwood", "pavelkataykin")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class GrutSetStatusForMigrCampaignsOneshot(
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val ytProvider: YtProvider,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val grutApiService: GrutApiService,
) : SimpleOneshot<GrutSetStatusForMigrCampaignsParam, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutSetStatusForMigrCampaignsOneshot::class.java)
        private val RESULT_TABLE_PATH = "//home/direct/test/uacconverter/GrutSetStatusForMigrCampaignsOneshot_${Environment.getCached().name.lowercase()}"
    }

    override fun validate(inputData: GrutSetStatusForMigrCampaignsParam) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }


    override fun execute(inputData: GrutSetStatusForMigrCampaignsParam, prevState: State?): State? {
        logger.info("Start from state=$prevState")
        val chunkSize = 1
        uacConverterYtRepository.createResultTableIfNotExists(RESULT_TABLE_PATH)
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val campaignIds = getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        if (campaignIds.isEmpty()) return null
        val campaignIdWithStatus = campaignIds[0]
        try {
            campaignIdWithStatus.let {
                grutTransactionProvider.runInTransaction {
                    val campaigns = grutUacCampaignService.getCampaigns(listOf(it.first.toString()))
                    if (campaigns.isEmpty()) {
                        val migrationResult = MigrationResult(campaignIdWithStatus.first, true, "Campaign not found in grut")
                        uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                        return@runInTransaction
                    }
                    val campaign = campaigns[0]
                    if (!campaign.spec.hasStatus()) {
                        val newSpec = campaign.spec.toBuilder().apply {
                            status = if (it.second == 1) Campaign.TCampaignSpec.ECampaignStatus.CS_CREATED
                            else Campaign.TCampaignSpec.ECampaignStatus.CS_DRAFT
                        }.build()
                        val campaignUpdated = Schema.TCampaign.newBuilder().apply {
                            meta = Schema.TCampaignMeta.newBuilder().setId(it.first).build()
                            spec = newSpec
                        }.build()
                        grutApiService.briefGrutApi.updateBriefFull(campaignUpdated)
                        val migrationResult = MigrationResult(campaignIdWithStatus.first, true, "")
                        uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                    } else {
                        val migrationResult = MigrationResult(campaignIdWithStatus.first, true, "Campaign already has status")
                        uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
                    }
                }
            }

        } catch (e: Exception) {
            logger.error("Got exception", e)
            val migrationResult = MigrationResult(campaignIdWithStatus.first, false, e.message)
            uacConverterYtRepository.writeResults(RESULT_TABLE_PATH, listOf(migrationResult))
        }

        return State(lastRow)
    }


    class InputTableRow : YtTableRow(listOf(CID, STATUS)) {
        companion object {
            private val CID = YtField("cid", Long::class.java)
            private val STATUS = YtField("ydbDirectCampaignStatus", Int::class.java)
        }

        val cid: Long?
            get() = valueOf(CID)

        val status: Int?
            get() = valueOf(STATUS)
    }

    fun getCampaignIdsFromYtTable(ytCluster: YtCluster, tablePath: String, startRow: Long, lastRow: Long): List<Pair<Long, Int>> {
        val campaignIds = mutableListOf<Pair<Long, Int>>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(YtTable(tablePath), {
                if (it.status != null) {
                    campaignIds.add(it.cid!! to it.status!!)
                }
            }, InputTableRow(), startRow, lastRow)
        return campaignIds
    }
}
