package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.uac.converter.UacBidModifiersConverter.toUacAdjustments
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toAssetGrut
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEAdCampaignType
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEAssetLinkStatus
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toECampaignStatus
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toRejectReason
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.model.direct_ad_group.DirectAdGroupStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.grut.api.AssetGrut
import ru.yandex.direct.core.grut.api.BriefAdGroup
import ru.yandex.direct.core.grut.api.BriefBanner
import ru.yandex.direct.core.grut.api.SitelinkAssetGrut
import ru.yandex.direct.core.grut.api.TextAssetGrut
import ru.yandex.direct.core.grut.api.TitleAssetGrut
import ru.yandex.direct.env.Environment
import ru.yandex.direct.oneshot.oneshots.uc.YdbGrutConverterYtRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.grut.objects.proto.AdGroup
import ru.yandex.grut.objects.proto.AssetLink
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.RejectReasons
import ru.yandex.grut.objects.proto.client.Schema

enum class YdbGrutMigrationStatus {
    OK,
    NOT_FOUND_IN_YDB,
    ERROR,
    MIGRATED_BUT_NOT_DELETED,

}

data class YdbGrutMigrationResult(
    val cid: Long,
    val ydbCampaignId: String? = null,
    val ydbDirectCampaignStatus: Int? = null,
    val status: YdbGrutMigrationStatus,
    val message: String? = null,
)

@Component
@Approvers("mspirit", "dimitrovsd", "elwood")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class GrutUacConverterOneshot(
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val uacConverterYtRepository: YdbGrutConverterYtRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
    private val uacYdbCampaignContentRepository: UacYdbCampaignContentRepository,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val uacYdbDirectAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val grutUacConverterRepository: GrutUacConverterRepository,
    private val uacYdbContentRepository: UacYdbContentRepository,
    private val bidModifierService: BidModifierService,
    private val clientGeoService: ClientGeoService,
) : ShardedOneshot<Param, State?> {
    private val chunkSizeProperty: PpcProperty<Int> = ppcPropertiesSupport.get(PpcPropertyNames.UC_UAC_CONVERTER_CHUNK_SIZE)

    companion object {
        private val logger = LoggerFactory.getLogger(UcUacConverterOneshot::class.java)
        private val resultTablePath = "//home/direct/test/uacconverter/GrutConvertionResult_${Environment.getCached().name.lowercase()}"
    }

    override fun validate(inputData: Param) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Param?, prevState: State?, shard: Int): State? {
        logger.info("Start from state=$prevState, shard=$shard")
        val chunkSize = chunkSizeProperty.getOrDefault(10)
        uacConverterYtRepository.createResultTableIfNotExists(resultTablePath)
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(inputData!!.ytCluster, inputData.tablePath, startRow, lastRow)
        if (cidsChunk.isEmpty()) return null
        val cidsChunkForShardWithClient = campaignTypedRepository.getSafely(shard, cidsChunk.distinct(), CommonCampaign::class.java).map { it.id to it.clientId }
        val campaignIdToBidModifiers = if (cidsChunkForShardWithClient.isNotEmpty()) {
            bidModifierService.getByCampaignIds(
                shard,
                cidsChunkForShardWithClient.map { it.first }.toSet(),
                setOf(BidModifierType.DEMOGRAPHY_MULTIPLIER, BidModifierType.GEO_MULTIPLIER),
                setOf(BidModifierLevel.CAMPAIGN)
            ).groupBy { it.campaignId }
        } else emptyMap()

        cidsChunkForShardWithClient.forEach {
            val campaignId = it.first
            val clientId = it.second
            val bidModifiers = campaignIdToBidModifiers.getOrDefault(campaignId, emptyList())

            val migrationResult = try {
                migrateCampaign(clientId, campaignId, bidModifiers)
            } catch (ex: Throwable) {
                val message = "Failed to migrate campaign $campaignId"
                logger.error(message, ex)
                YdbGrutMigrationResult(cid = campaignId, status = YdbGrutMigrationStatus.ERROR, message = message + " ${ex.message}")
            }
            uacConverterYtRepository.writeResults(resultTablePath, listOf(migrationResult))
            if (migrationResult.status != YdbGrutMigrationStatus.MIGRATED_BUT_NOT_DELETED) {
                if (migrationResult.status == YdbGrutMigrationStatus.ERROR) {
                    logger.error("Migration for campaign $campaignId failed")
                } else {
                    logger.warn("Campaign $campaignId wasn't migrate: ${migrationResult.message}")
                }
                return@forEach
            }
            logger.info("Delete uac campaign ${migrationResult.ydbCampaignId}(direct campaign id $campaignId from ydb")
            uacYdbDirectCampaignRepository.delete(migrationResult.ydbCampaignId!!)
            val finalMigrationResult = YdbGrutMigrationResult(cid = campaignId, ydbCampaignId = migrationResult.ydbCampaignId, ydbDirectCampaignStatus = migrationResult.ydbDirectCampaignStatus, status = YdbGrutMigrationStatus.OK)
            uacConverterYtRepository.writeResults(resultTablePath, listOf(finalMigrationResult))
            logger.info("Campaign $campaignId migration finished")
        }

        return if (cidsChunk.size < chunkSize) null else State(lastRow)
    }

    private fun migrateCampaign(clientId: Long, directCampaignId: Long, bidModifiers: List<BidModifier>): YdbGrutMigrationResult {
        val uacCampaignId = uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(listOf(directCampaignId))[directCampaignId]
        if (uacCampaignId == null) {
            val message = "Uac campaign for direct campaign id $directCampaignId not  found"
            logger.info(message)
            return YdbGrutMigrationResult(cid = directCampaignId, status = YdbGrutMigrationStatus.NOT_FOUND_IN_YDB, message = message)
        }

        val uacYdbCampaign = getCampaign(clientId, uacCampaignId, bidModifiers)
        val directCampaign = uacYdbDirectCampaignRepository.getDirectCampaignById(uacCampaignId)

        if (uacYdbCampaign == null || directCampaign == null) {
            val message = "Uac campaign $uacCampaignId with direct campaign id $directCampaignId not fount"
            logger.info(message)
            return YdbGrutMigrationResult(cid = directCampaignId, status = YdbGrutMigrationStatus.NOT_FOUND_IN_YDB, message = message)
        }

        if (uacYdbCampaign.advType != AdvType.CPM_BANNER && uacYdbCampaign.advType != AdvType.MOBILE_CONTENT) {
            val message = "Uac campaign $uacCampaignId with direct campaign id $directCampaignId has ${uacYdbCampaign.advType} which can't be converted"
            logger.info(message)
            return YdbGrutMigrationResult(directCampaignId, status = YdbGrutMigrationStatus.NOT_FOUND_IN_YDB, message = message)
        }

        val uacYdbCampaignContents = uacYdbCampaignContentRepository.getCampaignContents(uacCampaignId)
        val adGroups = uacYdbDirectAdGroupRepository.getDirectAdGroupsByCampaignId(uacCampaignId)
        val directAdGroupIdToAd =
            adGroups
                .flatMap {
                    getYdbBanners(it.id).map { ad ->
                        it.directAdGroupId to ad
                    }
                }

        val contentIds = uacYdbCampaignContents.mapNotNull { it.contentId }
        val contents = uacYdbContentRepository.getContents(contentIds)
            .associateBy { it.id }

        val grutAssets = uacYdbCampaignContents
            .map {
                val content = it.contentId?.let { contentId -> contents[contentId] }
                getAsset(clientId, it, content)
            }

        val (assetLinks, assetLinksStatuses) = getAssetLinksAndStatuses(uacYdbCampaignContents)
        val grutCampaign = Schema.TCampaign.newBuilder().apply {
            spec = UacGrutCampaignConverter.toCampaignSpec(uacYdbCampaign)
                .toBuilder()
                .setStatus(directCampaign.status.toECampaignStatus())
                .setBriefAssetLinks(assetLinks)
                .setBriefAssetLinksStatuses(assetLinksStatuses)
                .build()
            meta = Schema.TCampaignMeta.newBuilder().apply {
                id = directCampaignId
                this.clientId = clientId
                // время создания хранится в микросекундах, но домножаем всего на тысячу из-за бага YTORM-275
                creationTime = uacYdbCampaign.createdAt.atZone(DateTimeUtils.MSK).toEpochSecond() * 1_000
                campaignType = uacYdbCampaign.advType.toEAdCampaignType()
            }.build()
        }.build()
        val grutAdGroups = adGroups.map {
            BriefAdGroup(
                id = it.directAdGroupId,
                briefId = directCampaignId,
                status = it.status.toEAdGroupStats()
            )
        }
        val grutBanners = directAdGroupIdToAd
            .map {
                getBanner(directCampaignId, it.first, it.second)
            }
        grutUacConverterRepository.createClient(clientId)
        grutUacConverterRepository.createAssets(clientId, grutAssets)
        grutUacConverterRepository.createCampaign(grutCampaign)
        grutUacConverterRepository.createAdGroups(directCampaignId, grutAdGroups)
        grutUacConverterRepository.createBanners(directCampaignId, grutBanners)
        return YdbGrutMigrationResult(cid = directCampaignId, ydbCampaignId = uacCampaignId, ydbDirectCampaignStatus = directCampaign.status.id, status = YdbGrutMigrationStatus.MIGRATED_BUT_NOT_DELETED)
    }

    private fun getAssetLinksAndStatuses(campaignContents: List<UacYdbCampaignContent>): Pair<AssetLink.TAssetLinks, AssetLink.TAssetLinksStatuses> {
        val assetLinksWithStatuses = campaignContents
            .map { campaignContent ->
                val assetLink = AssetLink.TAssetLink.newBuilder().apply {
                    assetId = campaignContent.id.toIdLong()
                    val removeAtSeconds = campaignContent.removedAt?.atZone(DateTimeUtils.MSK)?.toEpochSecond()?.toInt()
                    if (removeAtSeconds != null) removeTime = removeAtSeconds
                    order = campaignContent.order
                    createTime = campaignContent.createdAt.atZone(DateTimeUtils.MSK)?.toEpochSecond()!!.toInt()
                    linkType = AssetLink.ELinkType.LT_NOT_SPECIFIED
                }.build()
                val assetLinkStatus = AssetLink.TAssetLinkStatus.newBuilder().apply {
                    assetId = campaignContent.id.toIdLong()
                    status = campaignContent.status.toEAssetLinkStatus()
                    linkType = AssetLink.ELinkType.LT_NOT_SPECIFIED
                    rejectReasons = RejectReasons.TRejectReasons.newBuilder().apply {
                        addAllRejectReasons(
                            campaignContent.rejectReasons?.map { it.toRejectReason() } ?: listOf()
                        )
                    }.build()

                }.build()
                assetLink to assetLinkStatus
            }
        val assetLinks = AssetLink.TAssetLinks.newBuilder().addAllLinks(assetLinksWithStatuses.map { it.first }).build()
        val assetLinksStatuses = AssetLink.TAssetLinksStatuses.newBuilder().addAllLinkStatuses(assetLinksWithStatuses.map { it.second }).build()
        return assetLinks to assetLinksStatuses
    }

    private fun getAsset(directClientId: Long, campaignContent: UacYdbCampaignContent, content: UacYdbContent?): AssetGrut {
        if (!(campaignContent.type == MediaType.TEXT || campaignContent.type == MediaType.TITLE || campaignContent.type == MediaType.SITELINK) && content == null) {
            throw RuntimeException("Content can't be null for type $campaignContent.type (campaign content id = ${campaignContent.id}") //todo
        }

        return when (campaignContent.type) {
            MediaType.TEXT -> TextAssetGrut(
                id = campaignContent.id.toIdLong(),
                clientId = directClientId,
                text = campaignContent.text!!,
            )
            MediaType.TITLE -> TitleAssetGrut(
                id = campaignContent.id.toIdLong(),
                clientId = directClientId,
                title = campaignContent.text!!,
            )
            MediaType.SITELINK -> SitelinkAssetGrut(
                id = campaignContent.id.toIdLong(),
                clientId = directClientId,
                title = campaignContent.sitelink!!.title,
                href = campaignContent.sitelink!!.href,
                description = campaignContent.sitelink!!.description,
            )
            else -> toAssetGrut(content!!)
        }
    }

    private fun getBanner(directCampaignId: Long, directAdGroupId: Long, ad: UacYdbDirectAd) =
        BriefBanner(
            id = ad.directAdId!!,
            adGroupId = directAdGroupId,
            briefId = directCampaignId,
            source = Banner.EBannerSource.BS_DIRECT,
            assetIds = listOfNotNull(
                ad.textContentId,
                ad.titleContentId,
                ad.directVideoContentId,
                ad.directImageContentId,
                ad.directHtml5ContentId,
                ad.directContentId,
            ).map { it.toIdLong() },
            assetLinksIds = listOf(),
            status = ad.status.toEBannerStatus()
        )

    private fun DirectAdStatus.toEBannerStatus(): Banner.TBannerSpec.EBannerStatus {
        return when (this) {
            DirectAdStatus.CREATED -> Banner.TBannerSpec.EBannerStatus.BSS_CREATED
            DirectAdStatus.ERROR_UNKNOWN -> Banner.TBannerSpec.EBannerStatus.BSS_ERROR_UNKNOWN
            DirectAdStatus.MODERATING -> Banner.TBannerSpec.EBannerStatus.BSS_MODERATING
            DirectAdStatus.ACTIVE -> Banner.TBannerSpec.EBannerStatus.BSS_ACTIVE
            DirectAdStatus.REJECTED -> Banner.TBannerSpec.EBannerStatus.BSS_REJECTED
            DirectAdStatus.ARCHIVED -> Banner.TBannerSpec.EBannerStatus.BSS_ARCHIVED
            DirectAdStatus.DELETED -> Banner.TBannerSpec.EBannerStatus.BSS_DELETED
            else -> Banner.TBannerSpec.EBannerStatus.BSS_NOT_SPECIFIED
        }
    }

    private fun DirectAdGroupStatus.toEAdGroupStats() =
        when (this) {
            DirectAdGroupStatus.CREATED -> AdGroup.TAdGroupSpec.EDirectAdGroupStatus.DAGS_CREATED
            else -> AdGroup.TAdGroupSpec.EDirectAdGroupStatus.DAGS_NOT_SPECIFIED
        }

    private fun getYdbBanners(ydbAdGroupId: String): List<UacYdbDirectAd> {
        val allYdbDirectAds = mutableListOf<UacYdbDirectAd>()

        var fromUacAdId = 0L
        while (true) {
            val ydbDirectAdsFromYdb = uacYdbDirectAdRepository
                .getByDirectAdGroupId(listOf(ydbAdGroupId), fromUacAdId, 990L)

            allYdbDirectAds.addAll(ydbDirectAdsFromYdb)
            if (ydbDirectAdsFromYdb.size < 990L) {
                break
            }

            // Здесь нельзя сравнивать и брать наибольший, т.к. значения id выше long.max - отрицательные
            fromUacAdId = ydbDirectAdsFromYdb.last().id.toIdLong()
        }
        return allYdbDirectAds
    }

    private fun getCampaign(clientId: Long, campaignId: String, bidModifiers: List<BidModifier>): UacYdbCampaign? {

        val uacYdbCampaign = uacYdbCampaignRepository.getCampaign(campaignId)

        if (uacYdbCampaign != null && uacYdbCampaign.advType == AdvType.MOBILE_CONTENT) {

            val geoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
            uacYdbCampaign.adjustments = toUacAdjustments(bidModifiers, geoTree)
        }

        return uacYdbCampaign
    }

}
