package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.grut.api.AssetGrut
import ru.yandex.direct.core.grut.api.BriefAdGroup
import ru.yandex.direct.core.grut.api.BriefBanner
import ru.yandex.direct.core.grut.api.ClientGrutModel
import ru.yandex.direct.core.grut.api.GrutApiBase
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.grut.objects.proto.client.Schema

@Component
class GrutUacConverterRepository(
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService
) {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutUacConverterRepository::class.java)
    }

    fun createClient(clientId: Long) {
        grutTransactionProvider.runRetryable(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val clientGrut = grutApiService.clientGrutDao.getClient(clientId)
            if (clientGrut == null) {
                grutApiService.clientGrutDao.createObjects(listOf(ClientGrutModel(Client().withId(clientId), listOf())))
            }
        }
    }

    fun createCampaign(campaign: Schema.TCampaign) {
        grutApiService.briefGrutApi.getBrief(campaign.meta.id, listOf("/meta/id"))
        val existingCampaign = grutApiService.briefGrutApi.getBrief(campaign.meta.id, listOf("/meta/id"))
        if (existingCampaign != null) {
            logger.info("Delete existing campaign ${existingCampaign.meta.id}")
            grutApiService.briefGrutApi.deleteObjects(listOf(existingCampaign.meta.id))
        }
        logger.info("Creating campaign ${campaign.meta.id}")
        grutApiService.briefGrutApi.createBrief(
            Schema.TCampaign.newBuilder().apply {
                meta = campaign.meta
                spec = campaign.spec
            }.build()
        )
    }

    fun createAdGroups(directCampaignId: Long, adGroups: List<BriefAdGroup>) {
        logger.info("Creating adGroups ${adGroups.map { it.id }} for campaign $directCampaignId")
        grutApiService.briefAdGroupGrutApi.createOrUpdateBriefAdGroups(adGroups)
    }

    fun createBanners(directCampaignId: Long, banners: List<BriefBanner>) {
        logger.info("Creating banners ${banners.map { it.id }} for campaign $directCampaignId")
        banners.chunked(500) { bannersChunk ->
            grutApiService.briefBannerGrutApi.createOrUpdateBriefBanners(bannersChunk)
        }
    }

    fun createAssets(clientId: Long, assets: List<AssetGrut>) {
        logger.info("Create assets ${assets.map { it.id }} for client $clientId")
        grutApiService.assetGrutApi.createObjects(assets)
    }
}
