package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import java.time.LocalDateTime
import java.time.LocalTime
import java.time.ZoneOffset
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.bidmodifier.AgeType
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile
import ru.yandex.direct.core.entity.bidmodifiers.Constants
import ru.yandex.direct.core.entity.bidmodifiers.container.ComplexBidModifierConverter
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService
import ru.yandex.direct.core.entity.campaign.model.BudgetDisplayFormat
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.AgePoint
import ru.yandex.direct.core.entity.uac.model.DeviceType
import ru.yandex.direct.core.entity.uac.model.DirectCampaignStatus
import ru.yandex.direct.core.entity.uac.model.Gender
import ru.yandex.direct.core.entity.uac.model.HolidaySettings
import ru.yandex.direct.core.entity.uac.model.LimitPeriodType
import ru.yandex.direct.core.entity.uac.model.Socdem
import ru.yandex.direct.core.entity.uac.model.TargetStatus
import ru.yandex.direct.core.entity.uac.model.TimeTarget
import ru.yandex.direct.core.entity.uac.model.UacCampaignOptions
import ru.yandex.direct.core.entity.uac.model.UacGoal
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAccountRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUserRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAccount
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbUser
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.libs.timetarget.HoursCoef
import ru.yandex.direct.libs.timetarget.WeekdayType
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.UacCampaignFetcher.Utils.convertDeviceTypes
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.UacCampaignFetcher.Utils.convertSocdem
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.UacCampaignFetcher.Utils.convertTimeTarget
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.UacCampaignFetcher.Utils.toLimitPeriodType
import ru.yandex.direct.rbac.RbacService

@Component
class UacCampaignFetcher(
    private val bidModifierService: BidModifierService,
    private val hyperGeoService: HyperGeoService,
    private val rbacService: RbacService,
    private val keywordRepository: KeywordRepository,
    private val ydbUserRepository: UacYdbUserRepository,
    private val ydbAccountRepository: UacYdbAccountRepository,
    private val clientGeoService: ClientGeoService,
) {
    fun fetchCampaign(shard: Int, campaignInfo: CampaignMigrationInfo): Pair<UacYdbCampaign, UacYdbDirectCampaign> {
        val uacCampaignId = UacYdbUtils.generateUniqueRandomId()
        val clientId = campaignInfo.campaign.clientId
        val campaign = campaignInfo.campaign
        val uid = rbacService.getChiefByClientId(ClientId.fromLong(clientId))
        val uidClient = UidAndClientId.of(uid, ClientId.fromLong(clientId))

        val account = getOrCreateClient(uidClient)
        val ydbCampaign = toUacYdbCampaign(shard, uacCampaignId, campaignInfo, account, campaignInfo.isDraftCampaign)

        val directCampaign = UacYdbDirectCampaign(
            id = uacCampaignId,
            directCampaignId = campaign.id,
            status = if (campaignInfo.isDraftCampaign) DirectCampaignStatus.DRAFT else DirectCampaignStatus.CREATED,
            syncedAt = LocalDateTime.now(),
            rejectReasons = null,
        )
        return ydbCampaign to directCampaign
    }


    private fun toUacYdbCampaign(
        shard: Int,
        uacCampId: String,
        campaignInfo: CampaignMigrationInfo,
        account: UacYdbAccount,
        isDraftCampaign: Boolean,
    ): UacYdbCampaign {
        val campaign = campaignInfo.campaign
        val banner = campaignInfo.banners.firstOrNull()
        val adGroup = campaignInfo.adGroupsInfo.firstOrNull()
        val strategy = campaign.strategy.strategyData
        val hyperGeo = if (adGroup?.adGroupId != null)
            hyperGeoService.getHyperGeoByAdGroupId(ClientId.fromLong(campaign.clientId), listOf(adGroup.adGroupId))[adGroup.adGroupId]
        else null
        val keywords = when {
            adGroup?.adGroupId == null -> null
            campaign.statusArchived -> keywordRepository.getArchivedKeywordsByAdGroupIds(shard, ClientId.fromLong(campaign.clientId),
                listOf(campaign.id), listOf(adGroup.adGroupId))[adGroup.adGroupId]
            else -> keywordRepository.getKeywordsByAdGroupIds(shard, ClientId.fromLong(campaign.clientId),
                listOf(adGroup.adGroupId))[adGroup.adGroupId]
        }

        val bidModifiers = if (adGroup?.adGroupId == null) null
        else ComplexBidModifierConverter.convertToComplexBidModifier(
            bidModifierService.getByAdGroupIds(shard, setOf(adGroup.adGroupId), Constants.ALL_TYPES, setOf(BidModifierLevel.ADGROUP))
        )
        val geo = when {
            adGroup?.geo != null -> {
                val geoTree = clientGeoService.getClientTranslocalGeoTree(campaignInfo.campaign.clientId)
                clientGeoService.convertForWeb(adGroup.geo, geoTree)
            }
            else -> null
        }

        return UacYdbCampaign(
            id = uacCampId,
            name = campaign.name,
            storeUrl = banner?.href ?: campaign.href,
            appId = null,
            regions = geo,
            minusRegions = null,
            trackingUrl = null,
            impressionUrl = null,
            createdAt = campaign.createTime,
            updatedAt = LocalDateTime.now(),
            startedAt = if (isDraftCampaign) null else LocalDateTime.ofEpochSecond(campaign.startDate.toEpochSecond(LocalTime.MIN, ZoneOffset.UTC), 0, ZoneOffset.UTC), // ???
            targetId = null,
            cpa = strategy.avgCpa,
            weekLimit = strategy.sum,
            account = account.id,
            options = campaign.budgetDisplayFormat?.let { UacCampaignOptions(it.toLimitPeriodType()) }
                ?: UacCampaignOptions(),
            skadNetworkEnabled = null,
            adultContentEnabled = null,
            targetStatus = if (campaign.statusShow) TargetStatus.STARTED else TargetStatus.STOPPED,
            contentFlags = mapOf(), // recalculate after creating
            advType = AdvType.TEXT,
            hyperGeoId = hyperGeo?.id,
            keywords = keywords?.map { it.phrase },
            minusKeywords = null,
            socdem = convertSocdem(bidModifiers?.demographyModifier),
            deviceTypes = convertDeviceTypes(bidModifiers?.desktopModifier, bidModifiers?.mobileModifier),
            inventoryTypes = null,
            goals = campaign.meaningfulGoals?.map { UacGoal(it.goalId, conversionValue = it.conversionValue) }
                ?: campaign.strategy.strategyData.goalId?.let { listOf(UacGoal(it)) },
            counters = campaign.metrikaCounters?.map { it.toInt() } ?: listOf(),
            permalinkId = banner?.permalinkId,
            phoneId = banner?.phoneId,
            calltrackingSettingsId = null,
            timeTarget = convertTimeTarget(campaign.timeTarget, campaign.timeZoneId),
            zenPublisherId = null,//text only
            strategy = null,//cpm only
            retargetingCondition = null,//cpm only
            videosAreNonSkippable = null,//cpm only
            brandSurveyId = null,//cpm only
            showsFrequencyLimit = null,//cpm only
            briefSynced = true,
            strategyPlatform = null,
            isEcom = null,//ecom only
            crr = null,//ecom only
            feedId = null,//ecom only
            feedFilters = null,//ecom only
            trackingParams = null,//ecom only
            cpmAssets = null,//cpm only
            campaignMeasurers = null, //cpm only
            uacBrandsafety = null, //cpm only
            uacDisabledPlaces = null, //cpm only
            recommendationsManagementEnabled = campaign.isRecommendationsManagementEnabled,
            priceRecommendationsManagementEnabled = campaign.isPriceRecommendationsManagementEnabled,
        )

    }

    private fun getOrCreateClient(uidClient: UidAndClientId): UacYdbAccount {
        val user = ydbUserRepository.getUserByUid(uidClient.uid)
        if (user == null) {
            val newUser = UacYdbUser(uid = uidClient.uid)
            ydbUserRepository.saveUser(newUser)
        }

        val account = ydbAccountRepository.getAccountByClientId(uidClient.clientId.asLong())
        if (account != null) {
            return account
        }
        val newAccount = UacYdbAccount(uid = uidClient.uid, directClientId = uidClient.clientId.asLong())
        ydbAccountRepository.saveAccount(newAccount)
        return newAccount
    }


    object Utils {
        fun convertSocdem(socdem: BidModifierDemographics?): Socdem {
            if (socdem == null || socdem.enabled != true || socdem.demographicsAdjustments.isEmpty())
                return Socdem(genders = Gender.values().toList(), AgePoint.AGE_0, AgePoint.AGE_INF, null, null)

            val genders: Set<Gender> = Gender.values().toSet() -
                socdem.demographicsAdjustments
                    .filter { it.age == null }
                    .map { Gender.valueOf(it.gender.toString()) }

            val bounds = mapOf(
                AgeType._0_17 to (AgePoint.AGE_0 to AgePoint.AGE_18),
                AgeType._18_24 to (AgePoint.AGE_18 to AgePoint.AGE_25),
                AgeType._25_34 to (AgePoint.AGE_25 to AgePoint.AGE_35),
                AgeType._35_44 to (AgePoint.AGE_35 to AgePoint.AGE_45),
                AgeType._45_54 to (AgePoint.AGE_45 to AgePoint.AGE_55),
                AgeType._55_ to (AgePoint.AGE_55 to AgePoint.AGE_INF),
            )
            val ages = socdem.demographicsAdjustments.mapNotNull { it.age }.toSet()
            val ageLower = bounds.entries.first { it.key !in ages }.value.first
            val ageUpper = bounds.entries.last { it.key !in ages }.value.second

            return Socdem(
                genders = genders.toList(),
                ageLower = ageLower,
                ageUpper = ageUpper,
                incomeLower = null,
                incomeUpper = null
            )
        }

        fun convertDeviceTypes(
            desktopModifier: BidModifierDesktop?,
            mobileModifier: BidModifierMobile?
        ): Set<DeviceType> =
            if (desktopModifier?.enabled == true) setOf(DeviceType.PHONE)
            else if (mobileModifier?.enabled == true) setOf(DeviceType.DESKTOP, DeviceType.TABLET)
            else setOf(DeviceType.ALL)

        internal fun BudgetDisplayFormat.toLimitPeriodType() =
            when (this) {
                BudgetDisplayFormat.MONTHLY -> LimitPeriodType.MONTH
                BudgetDisplayFormat.WEEKLY -> LimitPeriodType.WEEK
                else -> throw IllegalStateException("Incorrect BudgetDisplayFormat: $this")
            }

        fun convertTimeTarget(timeTarget: ru.yandex.direct.libs.timetarget.TimeTarget, timezoneId: Long): TimeTarget {
            val holidaySettings = if (WeekdayType.HOLIDAY in timeTarget.weekdayCoefs) {
                val hoursCoef = timeTarget.weekdayCoefs[WeekdayType.HOLIDAY]!!
                HolidaySettings(
                    startHour = (0..23).firstOrNull { hoursCoef.getCoefForHour(it) > 0 },
                    endHour = (0..23).lastOrNull { hoursCoef.getCoefForHour(it) > 0 }?.let { it + 1 },
                    rateCorrections = null,
                    show = (0..23).any { hoursCoef.getCoefForHour(it) > 0 }
                )
            } else null

            fun getDayCoefs(day: Int): List<Int> {
                val coefs = timeTarget.weekdayCoefs[WeekdayType.getById(day)] ?: HoursCoef()
                return (0..23).map { coefs.getCoefForHour(it) }
            }

            return TimeTarget(
                idTimeZone = timezoneId,
                useWorkingWeekends = WeekdayType.WORKING_WEEKEND in timeTarget.weekdayCoefs, // always true?
                enabledHolidaysMode = holidaySettings != null,
                holidaysSettings = holidaySettings,
                timeBoard = (1..7).map { getDayCoefs(it) }
            )
        }
    }
}
