package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import java.util.function.Function
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.util.RepositoryUtils.setFromDb
import ru.yandex.direct.common.util.RepositoryUtils.setToDb
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.env.Environment
import ru.yandex.direct.oneshot.oneshots.uc.UacConverterYtRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

data class FetchedCampaignInfo(
    val type: CampaignsType,
    val source: CampaignsSource,
    val opts: Set<String>,
)

/**
 * Ваншот для конвертации uc кампаний в ТГО
 * Нужен для тех компаний, которых не удалось мигрировать в ydb
 */
@Component
@Approvers("mspirit", "dimitrovsd")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class UcTgoConverterOneshot @Autowired constructor(
    private val dsl: DslContextProvider,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
) : ShardedOneshot<Param, State?> {
    companion object {
        private val logger = LoggerFactory.getLogger(UcTgoConverterOneshot::class.java)
        private const val CHUNK_SIZE = 100
        private val resultTablePath = "//home/direct/test/uacconverter/TgoConversionResult_${Environment.getCached().name.lowercase()}"
    }

    override fun validate(inputData: Param) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Param, prevState: State?, shard: Int): State? {
        logger.info("Start from state=$prevState, shard=$shard")
        uacConverterYtRepository.createResultTableIfNotExists(resultTablePath)
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + CHUNK_SIZE
        val campaignIdsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        val campaignIdsChunkShard = campaignTypedRepository.getSafely(shard, campaignIdsChunk.distinct(), CommonCampaign::class.java).map { it.id }
        campaignIdsChunkShard
            .asSequence()
            .map { convertCampaignToTgo(shard, it) }
            .forEach { uacConverterYtRepository.writeResults(resultTablePath, listOf(it)) }

        if (campaignIdsChunk.size < CHUNK_SIZE) {
            return null
        }
        return State(lastRow)
    }

    private fun convertCampaignToTgo(shard: Int, campaignId: Long): MigrationResult {
        if (uacYdbDirectCampaignRepository.getDirectCampaignByDirectCampaignId(campaignId) != null) {
            logger.info("Campaign $campaignId exists in ydb, skip it")
            return MigrationResult(campaignId, false, "Campaign $campaignId exists in ydb, skip it")
        }
        return try {
            dsl.ppc(shard).transactionResult { configuration ->
                val fetchedCampaignInfo: FetchedCampaignInfo? = configuration.dsl().select(CAMPAIGNS.TYPE, CAMPAIGNS.SOURCE, CAMPAIGNS.OPTS)
                    .from(CAMPAIGNS)
                    .where(CAMPAIGNS.CID.eq(campaignId))
                    .fetchOne { FetchedCampaignInfo(it.get(CAMPAIGNS.TYPE), it.get(CAMPAIGNS.SOURCE), setFromDb(it.get(CAMPAIGNS.OPTS))) }

                if (fetchedCampaignInfo == null) {
                    logger.error("Not found campaign $campaignId")
                    return@transactionResult MigrationResult(campaignId, false, "Not found text campaign with source uac $campaignId")
                }
                if (fetchedCampaignInfo.type == CampaignsType.text && fetchedCampaignInfo.source == CampaignsSource.direct) {
                    return@transactionResult MigrationResult(campaignId, true, "Already tgo campaign")
                }
                if (fetchedCampaignInfo.type != CampaignsType.text && fetchedCampaignInfo.source != CampaignsSource.uac) {
                    return@transactionResult MigrationResult(campaignId, false, "Campaign not suitable to convert")
                }
                val opts = fetchedCampaignInfo.opts.toMutableSet()
                opts.remove("is_universal")
                val newOpts = setToDb(opts, Function.identity())
                configuration.dsl()
                    .update(CAMPAIGNS)
                    .set(CAMPAIGNS.SOURCE, CampaignsSource.direct)
                    .set(CAMPAIGNS.OPTS, newOpts)
                    .where(CAMPAIGNS.CID.eq(campaignId))
                    .execute()
                return@transactionResult MigrationResult(campaignId, true, "")
            }
        } catch (e: Exception) {
            logger.error("Failed to convert campaign $campaignId from uc to tgo", e)
            MigrationResult(campaignId, false, "Failed to convert campaign $campaignId from uc to tgo")
        }
    }
}
