package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames.UC_UAC_CONVERTER_CHUNK_SIZE
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum
import ru.yandex.direct.core.entity.banner.model.TextBanner
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.model.direct_ad_group.DirectAdGroupStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAdGroup
import ru.yandex.direct.core.entity.uac.service.CampaignContentUpdateService
import ru.yandex.direct.core.entity.uac.service.UacCampaignService
import ru.yandex.direct.core.entity.uac.service.YdbUacClientService
import ru.yandex.direct.core.entity.user.repository.UserRepository
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.env.Environment
import ru.yandex.direct.oneshot.oneshots.uc.UacConverterYtRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.model.YtCluster

data class Param(
    val ytCluster: YtCluster,
    val tablePath: String,
    val writeResults: Boolean?
)

data class State(
    val lastRow: Long
)

data class AdGroupInfo(
    val adGroupId: Long,
    val geo: List<Long>?,
)

data class MigrationResult(
    val cid: Long,
    val isOk: Boolean,
    val message: String?,
)

data class CampaignMigrationInfo(
    val campaign: TextCampaign,
    val isDraftCampaign: Boolean,
    val banners: List<TextBanner>,
    val adGroupsInfo: List<AdGroupInfo>,
    val contents: ImageAndVideoContents?,
)

/**
 * Ваншот для создания uac-заявок для существующих "старых" uc-кампаний
 */
@Component
@Approvers("mspirit", "dimitrovsd")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class UcUacConverterOneshot @Autowired constructor(
    private val dsl: DslContextProvider,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val aggregatedStatusesViewService: AggregatedStatusesViewService,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
    private val uacYdbCampaignContentRepository: UacYdbCampaignContentRepository,
    private val uacAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val campaignContentUpdateService: CampaignContentUpdateService,
    private val uacCampaignService: UacCampaignService,
    private val campaignTypesRepository: CampaignTypedRepository,
    private val bannerTypedRepository: BannerTypedRepository,
    private val siteLinkSetRepository: SitelinkSetRepository,
    private val ucUacConverterRepository: UcUacConverterRepository,
    private val uacCampaignFetcher: UacCampaignFetcher,
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val uacClientService: YdbUacClientService,
    private val userRepository: UserRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : ShardedOneshot<Param, State?> {
    private val chunkSizeProperty: PpcProperty<Int> = ppcPropertiesSupport.get(UC_UAC_CONVERTER_CHUNK_SIZE)

    companion object {
        private val logger = LoggerFactory.getLogger(UcUacConverterOneshot::class.java)
        private val resultTablePath = "//home/direct/test/uacconverter/ConvertionResult_${Environment.getCached().name.lowercase()}"
    }

    override fun validate(inputData: Param) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(notNull())
                check(
                    Constraint.fromPredicate(
                        { uacConverterYtRepository.checkIfInputTableExists(inputData.ytCluster, it) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(inputData: Param, prevState: State?, shard: Int): State? {
        logger.info("Start from state=$prevState, shard=$shard")
        val chunkSize = chunkSizeProperty.getOrDefault(10)
        uacConverterYtRepository.createResultTableIfNotExists(resultTablePath)
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val cidsChunk = uacConverterYtRepository.getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        if (cidsChunk.isEmpty()) return null
        val cidsChunkForShard = campaignTypesRepository.getSafely(shard, cidsChunk.distinct(), CommonCampaign::class.java).map { it.id }

        val results = mutableListOf<MigrationResult>()
        val (notSyncedCampaigns, alreadySyncedCampaigns) = cidsChunkForShard.partition { uacYdbDirectCampaignRepository.getDirectCampaignByDirectCampaignId(it) == null }
        alreadySyncedCampaigns.forEach {
            logger.info("Campaign $it already has ydb brief")
            results.add(MigrationResult(it, true, "Already synced"))
        }
        val (campaignsMigrationInfo, campaignsWithNotFountContent) = getCampaignsMigrationInfo(shard, notSyncedCampaigns)

        campaignsWithNotFountContent.forEach { results.add(MigrationResult(it, isOk = false, "Not found some contents in ydb for campaign $it")) }

        results.addAll(
            campaignsMigrationInfo.map { campaignMigrationInfo ->
                var exception: Exception? = null
                try {
                    migrateCampaign(shard, campaignMigrationInfo)
                } catch (e: Exception) {
                    logger.error("Failed to migrate campaign ${campaignMigrationInfo.campaign.id}", e)
                    exception = e
                }
                MigrationResult(campaignMigrationInfo.campaign.id, exception == null, exception?.message)
            }
        )
        if (inputData.writeResults != false) {
            uacConverterYtRepository.writeResults(resultTablePath, results)
        }
        return if (cidsChunk.size < chunkSize) null else State(lastRow)
    }

    private fun migrateCampaign(shard: Int, campaignMigrationInfo: CampaignMigrationInfo) {
        deleteExistingInfo(campaignMigrationInfo)
        if (campaignMigrationInfo.adGroupsInfo.size > 1) {
            throw IllegalStateException("There is more than one adGroups for campaign ${campaignMigrationInfo.campaign.id}")
        }

        val activeBanners = campaignMigrationInfo.banners.filter { it.statusShow == true }
        if (activeBanners.size > 1) {
            throw IllegalStateException("There is more than one active banners for campaign ${campaignMigrationInfo.campaign.id}")
        }
        val directCampaignId = campaignMigrationInfo.campaign.id
        logger.info("Migrating campaign $directCampaignId")
        val clientId = campaignMigrationInfo.campaign.clientId

        val (ydbCampaign, directCampaign) = uacCampaignFetcher.fetchCampaign(shard, campaignMigrationInfo)
        val uacAdGroup = createAdGroup(ydbCampaign.id, campaignMigrationInfo)
        if (uacAdGroup != null) {
            createContentsAndAds(shard, ydbCampaign.id, uacAdGroup.id, campaignMigrationInfo)
        }

        uacYdbCampaignRepository.addCampaign(ydbCampaign)
        campaignContentUpdateService
            .updateCampaignContentsAndCampaignFlags(shard, listOf(directCampaignId), mapOf(directCampaignId to ydbCampaign.id))
        // связка с Директовой кампанией создается последним шагом, так как именно после этого фронт переключается на ydb¬
        uacYdbDirectCampaignRepository.saveDirectCampaign(directCampaign)
    }

    /**
     * Удалить всю созданную в ydb информацию по кампании. На случай, если в предыдущий шаг создание закончилось не успешно
     */
    private fun deleteExistingInfo(campaignInfo: CampaignMigrationInfo) {
        val existingBanners = uacYdbDirectAdRepository.getByDirectAdId(campaignInfo.banners.map { it.id })
        val directAdGroupId = campaignInfo.banners.map { it.adGroupId }.firstOrNull()
        val existingGroup = directAdGroupId?.let { uacAdGroupRepository.getDirectAdGroupByDirectAdGroupId(directAdGroupId) }
        val existingCampaignContents = existingGroup?.let { uacYdbCampaignContentRepository.getCampaignContents(existingGroup.directCampaignId) }
        val existingCampaign = existingGroup?.let { uacYdbCampaignRepository.getCampaign(existingGroup.directCampaignId) }
        // кампании и контенты удаляются первыми, если удалить первыми группы, то связь с campaign_id будет потеряна и в базе останется мусор
        if (existingCampaign != null) {
            logger.info("Campaign ${existingCampaign.id} for direct campaign ${campaignInfo.campaign.id} already exists, let's delete it")
            uacYdbCampaignRepository.delete(existingCampaign.id)
        }
        if (existingCampaignContents != null && existingCampaignContents.isNotEmpty()) {
            logger.info("Campaign contents ${existingCampaignContents.map { it.id }} for direct campaign ${campaignInfo.campaign.id} already exist, let's delete it")
            uacYdbCampaignContentRepository.delete(existingCampaignContents.map { it.id })
        }
        if (existingGroup != null) {
            logger.info("AdGroup ${existingGroup.id} for direct adGroup $directAdGroupId already exists, let's delete it")
            uacAdGroupRepository.delete(existingGroup.id)
        }
        if (existingBanners.isNotEmpty()) {
            logger.info("Banners ${existingBanners.map { it.id }} for direct banners ${campaignInfo.banners.map { it.id }} already exist, let's delete it")
            uacYdbDirectAdRepository.delete(existingBanners.map { it.id })
        }
    }

    private fun createAdGroup(uacCampaignId: String, campaignInfo: CampaignMigrationInfo): UacYdbDirectAdGroup? {
        val directAdGroupId = campaignInfo.adGroupsInfo.map { it.adGroupId }.firstOrNull()
        if (directAdGroupId == null) {
            return null
        }
        val directAdGroup = UacYdbDirectAdGroup(id = UacYdbUtils.generateUniqueRandomId(), directCampaignId = uacCampaignId, directAdGroupId = directAdGroupId, status = DirectAdGroupStatus.CREATED)
        uacAdGroupRepository.saveDirectAdGroup(directAdGroup)
        return directAdGroup
    }

    private fun createContentsAndAds(shard: Int, uacCampaignId: String, uacAdGroupId: String, campaignInfo: CampaignMigrationInfo) {
        val banners = campaignInfo.banners
        banners
            .map {
                val isBannerDeleted = it.statusArchived == true
                val titleContent = createTextContent(uacCampaignId, it.title, MediaType.TITLE, isBannerDeleted)
                val textContent = createTextContent(uacCampaignId, it.body, MediaType.TEXT, isBannerDeleted)
                createSiteLinksContents(shard, uacCampaignId, it, isBannerDeleted) //id сайтлинков не сохраняются в direct_ad
                val imageContent =
                    if (it.imageHash != null)
                        createVisualContent(uacCampaignId, campaignInfo.contents!!.imageContents[it.imageHash]!!, MediaType.IMAGE, isBannerDeleted)
                    else null

                val videoContent =
                    if (it.creativeId != null)
                        createVisualContent(uacCampaignId, campaignInfo.contents!!.videoContents[it.creativeId]!!, MediaType.VIDEO, isBannerDeleted)
                    else null
                val directAd = UacYdbDirectAd(
                    id = UacYdbUtils.generateUniqueRandomId(),
                    status = if (isBannerDeleted) DirectAdStatus.DELETED else DirectAdStatus.CREATED,
                    titleContentId = titleContent.id,
                    textContentId = textContent.id,
                    directContentId = null,
                    directImageContentId = imageContent?.id,
                    directVideoContentId = videoContent?.id,
                    directHtml5ContentId = null,
                    directAdId = it.id,
                    directAdGroupId = uacAdGroupId
                )
                uacYdbDirectAdRepository.saveDirectAd(directAd)
            }
    }

    private fun createTextContent(uacCampaignId: String, text: String, mediaType: MediaType, isDeleted: Boolean): UacYdbCampaignContent {
        val campaignContent = UacYdbCampaignContent(
            campaignId = uacCampaignId,
            type = mediaType,
            text = text,
            order = 0, //на кампании может быть только один активный текст и заголовок
            status = if (isDeleted) CampaignContentStatus.DELETED else CampaignContentStatus.CREATED, // calculate after all banners
            removedAt = if (isDeleted) LocalDateTime.now() else null
        )
        uacYdbCampaignContentRepository.addCampaignContents(listOf(campaignContent))
        return campaignContent
    }

    private fun createVisualContent(uacCampaignId: String, contentId: String, mediaType: MediaType, isDeleted: Boolean): UacYdbCampaignContent {
        val campaignContent = UacYdbCampaignContent(
            campaignId = uacCampaignId,
            type = mediaType,
            contentId = contentId,
            order = 0, //на кампании может быть только одна картинка или видео
            status = if (isDeleted) CampaignContentStatus.DELETED else CampaignContentStatus.CREATED, // calculate after all banners
            removedAt = if (isDeleted) LocalDateTime.now() else null
        )
        uacYdbCampaignContentRepository.addCampaignContents(listOf(campaignContent))
        return campaignContent
    }

    private fun createSiteLinksContents(shard: Int, uacCampaignId: String, banner: TextBanner, isDeleted: Boolean): List<UacYdbCampaignContent> {
        if (banner.sitelinksSetId == null) {
            return listOf()
        }
        val siteLinks = siteLinkSetRepository.getSitelinksBySetIds(shard, listOf(banner.sitelinksSetId)).values()
        val siteLinksContents = siteLinks.map {
            val convertedSiteLink = ru.yandex.direct.core.entity.uac.model.Sitelink(
                title = it.title,
                href = it.href,
                description = it.description
            )
            UacYdbCampaignContent(
                campaignId = uacCampaignId,
                type = MediaType.SITELINK,
                order = it.orderNum.toInt(),
                status = if (isDeleted) CampaignContentStatus.DELETED else CampaignContentStatus.CREATED, // calculate after all banners
                sitelink = convertedSiteLink,
            )
        }
        uacYdbCampaignContentRepository.addCampaignContents(siteLinksContents)
        return siteLinksContents
    }

    private fun getNotConvertedCids(shard: Int, campaignIds: Collection<Long>): List<Long> =
        dsl.ppc(shard)
            .select(CAMPAIGNS.CID)
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
            .and(CAMPAIGNS.SOURCE.eq(CampaignsSource.uac))
            .and(CAMPAIGNS.TYPE.eq(CampaignsType.text))
            .and(CAMPAIGNS.CID.`in`(campaignIds))
            .fetch(CAMPAIGNS.CID)

    // возвращает удачно полученные migrationInfo и список кампаний, для которых не нашлось конентов
    private fun getCampaignsMigrationInfo(shard: Int, cidsChunk: Collection<Long>): Pair<List<CampaignMigrationInfo>, List<Long>> {
        val cids = getNotConvertedCids(shard, cidsChunk)
        logger.info("got ${cids.size} campaigns to migrate")

        val campaigns = campaignTypesRepository.getSafely(shard, cids, TextCampaign::class.java)
        val bannersByCampaignIds = bannerTypedRepository.getBannersByCampaignIdsAndClass(shard, cids, TextBanner::class.java)
            .groupBy { it.campaignId }
        val clientIds = campaigns.map { it.clientId }
        val campaignIdsToStatusDraft = aggregatedStatusesViewService
            .getCampaignStatusesByIds(shard, cids.toSet()).mapValues { it.value.selfStatusObject?.get()?.status == GdSelfStatusEnum.DRAFT }
        val adGroupsInfo = getAdGroupsInfo(shard, campaigns)

        val result = mutableListOf<CampaignMigrationInfo>()
        val campaignsWithNotFountContents = mutableListOf<Long>()
        val accountIdsByClientIds = ucUacConverterRepository.getAccountIdsByClientIds(clientIds)
        campaigns.forEach { campaign ->
            logger.info("Getting migration info for campaign ${campaign.id}")
            val imageAndVideoContents = ImageAndVideoContents(mutableMapOf(), mutableMapOf())
            val banners = bannersByCampaignIds[campaign.id] ?: listOf()
            val accountId = accountIdsByClientIds[campaign.clientId]
                ?: getOrCreateAccount(shard, campaign.uid)

            var notFoundSomeContent = false
            banners.forEach { banner ->
                if (banner.imageHash != null && !imageAndVideoContents.imageContents.containsKey(banner.imageHash)) {
                    val imageContentId = ucUacConverterRepository.fetchAccountImage(accountId, banner.imageHash)
                    if (imageContentId != null) {
                        imageAndVideoContents.imageContents[banner.imageHash] = imageContentId
                    } else {
                        notFoundSomeContent = true
                    }
                }

                if (banner.creativeId != null && !imageAndVideoContents.videoContents.containsKey(banner.creativeId)) {
                    val videoContentId = ucUacConverterRepository.fetchAccountVideo(accountId, banner.creativeId)
                    if (videoContentId != null) {
                        imageAndVideoContents.videoContents[banner.creativeId] = videoContentId
                    } else {
                        notFoundSomeContent = true
                    }
                }
            }

            val adGroupInfo = adGroupsInfo[campaign.id] ?: listOf()
            if (!notFoundSomeContent) {
                result.add(CampaignMigrationInfo(
                    campaign = campaign,
                    isDraftCampaign = campaignIdsToStatusDraft[campaign.id] ?: false,
                    banners = banners,
                    adGroupsInfo = adGroupInfo,
                    contents = imageAndVideoContents)
                )
            } else {
                campaignsWithNotFountContents.add(campaign.id)
                logger.error("Not found some contents in ydb for campaign ${campaign.id}. Skip this campaign")
            }

        }
        return result to campaignsWithNotFountContents
    }

    private fun getAdGroupsInfo(shard: Int, campaigns: Collection<TextCampaign>): Map<Long, MutableList<AdGroupInfo>> {
        val result = mutableMapOf<Long, MutableList<AdGroupInfo>>()
        dsl.ppc(shard)
            .select(PHRASES.CID, PHRASES.PID, PHRASES.GEO)
            .from(PHRASES)
            .where(PHRASES.CID.`in`(campaigns.map { it.id }))
            .fetch()
            .forEach {
                result.computeIfAbsent(it.get(PHRASES.CID)) { mutableListOf() }.add(AdGroupInfo(it.get(PHRASES.PID), AdGroupMappings.geoFromDb(it.get(PHRASES.GEO))))
            }
        return result
    }

    private fun getOrCreateAccount(shard: Int, uid: Long): String {
        val user = userRepository.fetchByUids(shard, listOf(uid))[0]
        val accountId = uacClientService.getOrCreateClient(user, user)
        logger.info("Got or created account $accountId for client ${user.clientId} uid $uid")
        return accountId
    }
}
