package ru.yandex.direct.oneshot.oneshots.uc.uacconverter

import java.time.Duration
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.ACCOUNT
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.AccountTable
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CONTENT
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.ContentTable
import ru.yandex.direct.utils.JsonUtils.fromJson
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.expression.JsonExpression.jsonValue
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.client.YdbClient


data class ImageAndVideoContents(
    val imageContents: MutableMap<String, String>,
    val videoContents: MutableMap<Long, String>,
)

@Repository
class UcUacConverterRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath
) {


    fun getAccountIdsByClientIds(clientIds: Collection<Long>): Map<Long, String> {
        val accountClientIndex = ACCOUNT.withIndex(ACCOUNT.ACCOUNT_CLIENT_ID_INDEX) as AccountTable
        val accountsQueryAndParams = select(accountClientIndex.DIRECT_CLIENT_ID, accountClientIndex.ID)
            .from(accountClientIndex)
            .where(accountClientIndex.DIRECT_CLIENT_ID.`in`(clientIds.distinct()))
            .queryAndParams(path)
        val accountsResultSet = ydbClient.executeOnlineRoQuery(accountsQueryAndParams, false)
            .getResultSet(0)

        val accountIdByClientIdMap = mutableMapOf<Long, String>()
        while (accountsResultSet.next()) {
            val clientId = accountsResultSet.getValueReader(accountClientIndex.DIRECT_CLIENT_ID).uint64
            val accountId = accountsResultSet.getValueReader(accountClientIndex.ID).uint64.toIdString()
            accountIdByClientIdMap[clientId] = accountId
        }
        return accountIdByClientIdMap
    }

    fun getAllClientContents(clientIds: Collection<Long>): Map<Long, ImageAndVideoContents> {
        val accountClientIndex = ACCOUNT.withIndex(ACCOUNT.ACCOUNT_CLIENT_ID_INDEX) as AccountTable
        val accountsQueryAndParams = select(accountClientIndex.DIRECT_CLIENT_ID, accountClientIndex.ID)
            .from(accountClientIndex)
            .where(accountClientIndex.DIRECT_CLIENT_ID.`in`(clientIds.distinct()))
            .queryAndParams(path)
        val accountsResultSet = ydbClient.executeOnlineRoQuery(accountsQueryAndParams, false)
            .getResultSet(0)

        val clientIdByAccountId = mutableMapOf<String, Long>()
        while (accountsResultSet.next()) {
            val clientId = accountsResultSet.getValueReader(accountClientIndex.DIRECT_CLIENT_ID).uint64
            val accountId = accountsResultSet.getValueReader(accountClientIndex.ID).uint64.toIdString()
            clientIdByAccountId[accountId] = clientId
        }

        val clientsContents = mutableMapOf<Long, ImageAndVideoContents>()
        var nextContentId: String? = "0"
        do {
            val fetchResult = fetchContent(clientIdByAccountId, nextContentId!!)
            clientsContents.putAll(fetchResult.clientsContents)
            nextContentId = fetchResult.maxContentId
        } while (nextContentId != null)
        return clientsContents

    }

    fun fetchAccountImage(accountId: String, imageHash: String): String? {
        val contentAccountIdIndex = CONTENT.withIndex(CONTENT.ACCOUNT_ID_INDEX) as ContentTable
        val queryAndParams = select(contentAccountIdIndex.ID)
            .from(contentAccountIdIndex)
            .where(contentAccountIdIndex.ACCOUNT_ID.eq(accountId.toIdLong())
                .and(contentAccountIdIndex.TYPE.eq(MediaType.IMAGE.id))
                .and(contentAccountIdIndex.DIRECT_IMAGE_HASH.eq(imageHash)))
            .limit(1)
            .queryAndParams(path)
        val resultSet = ydbClient.executeOnlineRoQuery(queryAndParams, true, Duration.ofMinutes(5))
            .getResultSet(0)
        if (resultSet.next()) {
            return resultSet.getValueReader(contentAccountIdIndex.ID).uint64.toIdString()
        }
        return null

    }

    fun fetchAccountVideo(accountId: String, contentId: Long): String? {
        val contentAccountIdIndex = CONTENT.withIndex(CONTENT.ACCOUNT_ID_INDEX) as ContentTable
        val queryAndParams = select(contentAccountIdIndex.ID)
            .from(contentAccountIdIndex)
            .where(contentAccountIdIndex.ACCOUNT_ID.eq(accountId.toIdLong())
                .and(contentAccountIdIndex.TYPE.eq(MediaType.VIDEO.id))
                .and(jsonValue(contentAccountIdIndex.META, "$.creative_id").eq(contentId.toString())))
            .limit(1)
            .queryAndParams(path)
        val resultSet = ydbClient.executeOnlineRoQuery(queryAndParams, true, Duration.ofMinutes(5))
            .getResultSet(0)
        if (resultSet.next()) {
            return resultSet.getValueReader(contentAccountIdIndex.ID).uint64.toIdString()
        }
        return null

    }

    private fun fetchContent(clientIdsByAccountIds: Map<String, Long>, minContentId: String): FetchContentsResult {
        val accountIds = clientIdsByAccountIds.keys
        val contentAccountIdIndex = CONTENT.withIndex(CONTENT.ACCOUNT_ID_INDEX) as ContentTable
        val contentsQueryAndParams =
            select(
                contentAccountIdIndex.ACCOUNT_ID,
                contentAccountIdIndex.ID,
                contentAccountIdIndex.DIRECT_IMAGE_HASH,
                contentAccountIdIndex.META
            )
                .from(contentAccountIdIndex)
                .where(
                    contentAccountIdIndex.ACCOUNT_ID.`in`(accountIds.toIdsLong())
                        .and(contentAccountIdIndex.ID.gt(minContentId.toIdLong()))
                        .and(contentAccountIdIndex.TYPE.`in`(MediaType.IMAGE.id, MediaType.VIDEO.id)))
                .orderBy(contentAccountIdIndex.ID)
                .queryAndParams(path)
        val clientsContents = mutableMapOf<Long, ImageAndVideoContents>()
        val contentsResultSet = ydbClient.executeOnlineRoQuery(contentsQueryAndParams, false, Duration.ofMinutes(5))
            .getResultSet(0)
        val isTruncated = contentsResultSet.isTruncated
        var lastContentId: String? = null
        while (contentsResultSet.next()) {
            val accountId = contentsResultSet.getValueReader(contentAccountIdIndex.ACCOUNT_ID).uint64.toIdString()
            val contentId = contentsResultSet.getValueReader(contentAccountIdIndex.ID).uint64.toIdString()
            val imageHash = contentsResultSet.getValueReaderOrNull(contentAccountIdIndex.DIRECT_IMAGE_HASH)?.utf8
            val creativeId = fromJson(contentsResultSet.getValueReader(contentAccountIdIndex.META).json)["creative_id"]?.longValue()
            val clientId = clientIdsByAccountIds[accountId]!!
            val currentClientContents = clientsContents.computeIfAbsent(clientId) {
                ImageAndVideoContents(mutableMapOf(), mutableMapOf())
            }
            if (imageHash != null) {
                currentClientContents.imageContents[imageHash] = contentId
            } else if (creativeId != null) {
                currentClientContents.videoContents[creativeId] = contentId
            }
            lastContentId = contentId
        }
        return FetchContentsResult(clientsContents, if (!isTruncated) null else lastContentId)
    }

    data class FetchContentsResult(
        val clientsContents: Map<Long, ImageAndVideoContents>,
        val maxContentId: String?
    )
}
