package ru.yandex.direct.oneshot.oneshots.update_metrika_conversion_sources

import org.jetbrains.annotations.Nullable
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignClientIdFilter
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignTypesExceptFilter
import ru.yandex.direct.core.entity.client.container.ClientsQueryFilter
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.conversionsource.service.ConversionCenterMetrikaGoalsService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.multipleConditionFilter
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject
import java.util.Collections.max

private const val CLIENTS_FOR_ITERATION_NUMBER_LIMIT = 500
private const val SPECIFIED_CLIENTS_FOR_ITERATION_NUMBER_LIMIT = 100

@Component
@Multilaunch
@Approvers("zakhar", "buhter", "munira")
class UpdateMetrikaConversionSourceOneshot(
    private val clientRepository: ClientRepository,
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val conversionCenterMetrikaGoalsService: ConversionCenterMetrikaGoalsService,
) : ShardedOneshot<InputData, State?> {

    override fun validate(inputData: InputData) = validateObject(inputData) {
        listProperty(inputData::clientIds)
            .check(maxListSize(SPECIFIED_CLIENTS_FOR_ITERATION_NUMBER_LIMIT))
            .checkEach(validId(), When.isValid())
            .checkEach(
                Constraint.fromPredicate(
                    { clientId -> shardHelper.isExistentClientId(clientId!!) }, CommonDefects.objectNotFound()
                ), When.isValid()
            )
    }

    @Nullable
    override fun execute(inputData: InputData, prevState: State?, shard: Int): State? {
        var clientIds = inputData.clientIds

        if (clientIds == null) {
            if (prevState == null) {
                return State(0)
            }

            val limitedNextClientsFilter =
                ClientsQueryFilter.getLimitedNextClients(ClientId.fromLong(prevState.lastClientId), CLIENTS_FOR_ITERATION_NUMBER_LIMIT)
            clientIds = clientRepository.getClientsByFilter(shard, limitedNextClientsFilter).map { it.id }

            if (clientIds.isEmpty()) {
                return null
            }
        }

        clientIds.forEach { id ->
            val clientId = ClientId.fromLong(id)
            val dslContext = dslContextProvider.ppc(shard)
            val campaigns = campaignTypedRepository.getTypedCampaigns(dslContext,
                multipleConditionFilter(campaignClientIdFilter(clientId), campaignTypesExceptFilter(setOf(CampaignType.MCB))))
            conversionCenterMetrikaGoalsService.addCampaignGoalsToConversionCenter(clientId, campaigns, true)
        }

        return if (inputData.clientIds == null) State(max(clientIds)) else null
    }
}

data class InputData(
    val clientIds: List<Long>?
)

data class State(
    val lastClientId: Long
)
