package ru.yandex.direct.oneshot.oneshots.updateconversionlevelforallmetrikagoals;

import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.autobudget.model.ConversionLevelToUpdate;
import ru.yandex.direct.core.entity.autobudget.model.GoalIdWithConversionLevel;
import ru.yandex.direct.core.entity.autobudget.ytmodels.generated.YtGoalQualityRow;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.JooqModelToDbFieldValuesMapper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;

import static ru.yandex.direct.core.entity.autobudget.ytmodels.generated.YtDbTables.GOALQUALITY;
import static ru.yandex.direct.dbschema.ppcdict.tables.MetrikaGoals.METRIKA_GOALS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Component
@Multilaunch
@Approvers({"pashkus", "ssdmitriev"})
public class UpdateConversionLevelForAllMetrikaGoalsOneshot implements SimpleOneshot<InputData, Void> {
    private static final Logger logger = LoggerFactory.getLogger(UpdateConversionLevelForAllMetrikaGoalsOneshot.class);

    private static final int YT_CHUNK_SIZE = 10_000;
    private static final Set<String> AVAILABLE_CLUSTERS = Set.of("hahn", "arnold");
    private static final int UPDATE_CHUNK_SIZE = 1000;

    private static final JooqModelToDbFieldValuesMapper<GoalIdWithConversionLevel> MAPPER = getMapper();

    private final DslContextProvider dslContextProvider;
    private final YtProvider ytProvider;

    public UpdateConversionLevelForAllMetrikaGoalsOneshot(DslContextProvider dslContextProvider,
                                                          YtProvider ytProvider) {
        this.dslContextProvider = dslContextProvider;
        this.ytProvider = ytProvider;
    }

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        ItemValidationBuilder<InputData, Defect> ivb = ItemValidationBuilder.of(inputData);
        ivb.item(inputData.getCluster(), "cluster")
                .check(notNull())
                .check(notBlank())
                .check(inSet(AVAILABLE_CLUSTERS));
        return ivb.getResult();
    }

    @Override
    public Void execute(InputData inputData, Void prevState) {
        AtomicInteger changedCount = new AtomicInteger();

        YtCluster cluster = YtCluster.parse(inputData.getCluster());
        YtOperator operator = ytProvider.getOperator(cluster);

        logger.info("Starting metrika_goals.conversion_level update on cluster {}", cluster.getName());

        operator.readTableSnapshot(GOALQUALITY, new YtGoalQualityRow(),
                UpdateConversionLevelForAllMetrikaGoalsOneshot::toGoalIdWithConversionLevel,
                goals -> changedCount.addAndGet(updateConversionLevel(goals)), YT_CHUNK_SIZE);

        logger.info("{} metrika_goals records updated", changedCount.get());

        return null;
    }

    private int updateConversionLevel(List<GoalIdWithConversionLevel> goalsToUpdate) {
        List<AppliedChanges<GoalIdWithConversionLevel>> changes = StreamEx.of(goalsToUpdate)
                .map(goalToUpdate -> new ModelChanges<>(goalToUpdate.getId(), GoalIdWithConversionLevel.class)
                        .process(goalToUpdate.getConversionLevelToUpdate(),
                                GoalIdWithConversionLevel.CONVERSION_LEVEL_TO_UPDATE)
                        .applyTo(new GoalIdWithConversionLevel().withId(goalToUpdate.getId()))
                )
                .toList();

        return updateConversionLevel(dslContextProvider.ppcdict(), changes);
    }

    private int updateConversionLevel(DSLContext context, List<AppliedChanges<GoalIdWithConversionLevel>> changes) {
        return StreamEx.ofSubLists(changes, UPDATE_CHUNK_SIZE)
                .map(chunk -> new UpdateHelper<>(context, METRIKA_GOALS.GOAL_ID)
                        .processUpdateAll(MAPPER, chunk)
                        .execute())
                .peek(count -> logger.info("updated {} records", count))
                .foldLeft(0, Integer::sum);
    }

    private static GoalIdWithConversionLevel toGoalIdWithConversionLevel(YtGoalQualityRow goalQualityRow) {
        return new GoalIdWithConversionLevel()
                .withId(goalQualityRow.getGoalID())
                .withConversionLevelToUpdate(getConversionLevel(goalQualityRow));
    }

    private static ConversionLevelToUpdate getConversionLevel(YtGoalQualityRow goalQuality) {
        return goalQuality.getIsPerfectGoal() == 1 ? ConversionLevelToUpdate.PERFECT :
                goalQuality.getIsGoodGoal() == 1 ? ConversionLevelToUpdate.GOOD :
                        goalQuality.getIsNotBadGoal() == 1 ? ConversionLevelToUpdate.SOSO : ConversionLevelToUpdate.BAD;
    }

    private static JooqModelToDbFieldValuesMapper<GoalIdWithConversionLevel> getMapper() {
        return JooqMapperWithSupplierBuilder.builder(GoalIdWithConversionLevel::new)
                .map(property(GoalIdWithConversionLevel.ID, METRIKA_GOALS.GOAL_ID))
                .map(convertibleProperty(
                        GoalIdWithConversionLevel.CONVERSION_LEVEL_TO_UPDATE, METRIKA_GOALS.CONVERSION_LEVEL,
                        ConversionLevelToUpdate::fromSource,
                        ConversionLevelToUpdate::toSource))
                .build();
    }
}
