package ru.yandex.direct.oneshot.oneshots.updategeotargeting

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.regions.GeoTreeFactory
import ru.yandex.direct.ytwrapper.client.YtProvider

/**
 * Ваншот для обновления геотаргетингов на группах.
 * ID групп для обновления получает из статической таблицы YT.
 * Записывает затронутые группы на переотправку в БК в очередь bs_resync_queue.
 */
@Component
@Multilaunch
@Approvers("maxlog")
open class UpdateAdGroupGeoTargetingOneshot @Autowired constructor(
    ytProvider: YtProvider,
    geoTreeFactory: GeoTreeFactory,
    private val adGroupRepository: AdGroupRepository,
    private val campaignRepository: CampaignRepository,
    private val bsResyncService: BsResyncService
) : UpdateGeoTargetingOneshot(ytProvider, geoTreeFactory) {

    override fun executeInternal(inputData: UpdateGeoInputData, start: Long, end: Long): Int {
        val adGroups = readFromYtTable(inputData, AdGroupWithShardTableRow(), start, end)
        val updatedByShard = adGroups
            .groupBy { it.shard }
            .mapValues { (shard, groups) ->
                executeInternalSharded(inputData, shard, groups.map { it.pid })
            }
        return updatedByShard.values.sum()
    }

    protected open fun executeInternalSharded(inputData: UpdateGeoInputData, shard: Int, adGroupIds: List<Long>): Int {
        val adGroupToNewGeo = adGroupRepository
            .getAdGroups(shard, adGroupIds)
            .associateWith { updatedGeoTargeting(inputData, it.geo.toSet()) }
            .filter { (adGroup, geo) -> adGroup.geo.toSet() != geo }
        adGroupToNewGeo.forEach { (group, geo) ->
            logger.info("Updating ad group id = ${group.id}: old geo = ${group.geo}, new geo = $geo")
        }
        updateAdGroupGeoTargeting(shard, adGroupToNewGeo)
        logger.info("Sending ad groups to bs resync: ids = ${adGroupToNewGeo.keys.map { it.id }}")
        sendAdGroupsToResync(shard, adGroupToNewGeo.keys)
        return adGroupToNewGeo.size
    }

    protected fun updateAdGroupGeoTargeting(
        shard: Int,
        adGroupToNewGeo: Map<AdGroup, Set<Long>>
    ) {
        val modelChanges = adGroupToNewGeo.map { (group, geo) ->
            ModelChanges.build(group.id, AdGroup::class.java, AdGroup.GEO, geo.toList()).applyTo(group)
        }
        modelChanges.chunked(WRITE_SIZE)
            .forEach { adGroupRepository.updateAdGroupsCommonFields(shard, it) }
    }

    protected fun sendAdGroupsToResync(shard: Int, adGroups: Collection<AdGroup>) {
        val campaigns = campaignRepository.getCampaignsMap(shard, adGroups.map { it.campaignId })
        val resyncItems = adGroups
            .filter { campaigns.containsKey(it.campaignId) }
            .filter { needResync(campaigns[it.campaignId]!!) }
            .map { BsResyncItem(getResyncPriority(campaigns[it.campaignId]!!), it.campaignId, null, it.id) }
        bsResyncService.addObjectsToResync(resyncItems)
    }
}
