package ru.yandex.direct.oneshot.oneshots.updategeotargeting

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.regions.GeoTreeFactory
import ru.yandex.direct.ytwrapper.client.YtProvider

/**
 * Ваншот для обновления геотаргетингов на кампаниях.
 * ID кампаний для обновления получает из статической таблицы YT.
 * Записывает затронутые кампании на переотправку в БК в очередь bs_resync_queue.
 */
@Component
@Multilaunch
@Approvers("maxlog")
class UpdateCampaignGeoTargetingOneshot @Autowired constructor(
    ytProvider: YtProvider,
    geoTreeFactory: GeoTreeFactory,
    private val campaignRepository: CampaignRepository,
    private val bsResyncService: BsResyncService,
) : UpdateGeoTargetingOneshot(ytProvider, geoTreeFactory) {

    override fun executeInternal(inputData: UpdateGeoInputData, start: Long, end: Long): Int {
        val campaigns = readFromYtTable(inputData, CampaignWithShardTableRow(), start, end)
        val updatedByShard = campaigns
            .groupBy { it.shard }
            .mapValues { (shard, campaigns) ->
                executeInternalSharded(inputData, shard, campaigns.map { it.campaignId })
            }
        return updatedByShard.values.sum()
    }

    private fun executeInternalSharded(inputData: UpdateGeoInputData, shard: Int, campaignIds: List<Long>): Int {
        val campaignToNewGeo = campaignRepository
            .getCampaigns(shard, campaignIds)
            .associateWith { updatedGeoTargeting(inputData, toLongSet(it.geo)) }
            .filter { (camp, geo) -> toLongSet(camp.geo) != geo }
        campaignToNewGeo.forEach { (campaign, geo) ->
            logger.info("Updating campaign id = ${campaign.id}: old geo = ${campaign.geo}, new geo = $geo")
        }
        updateCampaignGeoTargeting(shard, campaignToNewGeo)
        logger.info("Sending campaigns to bs resync: ids = ${campaignToNewGeo.keys.map { it.id }}")
        sendCampaignsToResync(campaignToNewGeo.keys)
        return campaignToNewGeo.size
    }

    private fun updateCampaignGeoTargeting(shard: Int, campaignToGeo: Map<Campaign, Set<Long>>) {
        val modelChanges = campaignToGeo.map { (camp, geo) ->
            ModelChanges.build(camp.id, Campaign::class.java, Campaign.GEO, toIntSet(geo)).applyTo(camp)
        }
        campaignRepository.updateCampaigns(shard, modelChanges)
    }

    // проверить приоритет отправки на синхронизацию
    private fun sendCampaignsToResync(campaigns: Collection<Campaign>) {
        val resyncItems = campaigns
            .filter { needResync(it) }
            .map { BsResyncItem(getResyncPriority(it), it.id, null, null) }
        bsResyncService.addObjectsToResync(resyncItems)
    }
}
