package ru.yandex.direct.oneshot.oneshots.updategeotargeting

import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.oneshot.base.SimpleYtOneshot
import ru.yandex.direct.oneshot.base.YtInputData
import ru.yandex.direct.oneshot.base.YtState
import ru.yandex.direct.regions.GeoTree
import ru.yandex.direct.regions.GeoTreeFactory
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.check
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtOperator
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import kotlin.math.min

// приоритеты отправки на синхронизацию в БК (скопированы из perl/protected/one-shot/relocate-zab-bu-geo.pl)
const val BS_RESYNC_PRIORITY_HIGH = 94L
const val BS_RESYNC_PRIORITY_LOW = 46L

data class UpdateRegionParam(
    var regionId: Long,
    var oldParentId: Long,
    var newParentId: Long
)

data class UpdateGeoInputData(
    var regionUpdates: List<UpdateRegionParam>
) : YtInputData()

abstract class ConvertibleTableRow<T>(
    columns: List<YtField<out Any>>
) : YtTableRow(columns) {
    abstract fun convert(): T
}

/**
 * Базовый ваншот для обновления геотаргетингов на кампаниях/группах
 * при внесении изменений в геодерево.
 *
 * Обрабатывает только ситуации переноса листового региона.
 */
abstract class UpdateGeoTargetingOneshot constructor(
    ytProvider: YtProvider,
    private val geoTreeFactory: GeoTreeFactory
) : SimpleYtOneshot<UpdateGeoInputData, YtState>(ytProvider) {

    protected val geoTree: GeoTree
        get() = geoTreeFactory.globalGeoTree

    companion object {
        val logger = LoggerFactory.getLogger(UpdateGeoTargetingOneshot::class.java)
        const val CHUNK_SIZE = 10_000L
        const val WRITE_SIZE = 1_000
    }

    override fun validate(inputData: UpdateGeoInputData): ValidationResult<UpdateGeoInputData, Defect<Any>> {
        return validateYt(inputData).result.merge(validateRegionUpdates(inputData).result)
    }

    private fun validateRegionUpdates(
        inputData: UpdateGeoInputData
    ): ItemValidationBuilder<UpdateGeoInputData, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        vb.list(inputData.regionUpdates, "regionUpdates")
            .checkEachBy { regionUpdate -> validateRegion(regionUpdate).result }
        return vb
    }

    private fun validateRegion(inputData: UpdateRegionParam): ItemValidationBuilder<UpdateRegionParam, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        vb.item(inputData.oldParentId, "oldParentId")
            .check(CommonDefects.objectNotFound()) { geoTree.hasRegion(it) }
        vb.item(inputData.newParentId, "newParentId")
            .check(CommonDefects.objectNotFound()) { geoTree.hasRegion(it) }
        vb.item(inputData.regionId, "regionId")
            .check(CommonDefects.objectNotFound()) { geoTree.hasRegion(it) }
            .check(CommonDefects.invalidValue()) { geoTree.getChildren(it).isEmpty() }
        return vb
    }

    /**
     * В ваншоты стоит добавить проверку, чтобы не трогать геотаргетинги,
     * которые пользователь успел обновить вперед ваншота.
     * А именно, если пользователь внес изменения в геотаргетинги после переключения геодерева,
     * но до миграции геотаргетингов ваншотом,
     * стоит считать такое обновление актуальным и согласованным с новым геодеревом
     * и не изменять геотаргетинг в ваншоте.
     *
     * Подробнее - [DIRECT-167692](https://st.yandex-team.ru/DIRECT-167692)
     * и [DIRECT-166212](https://st.yandex-team.ru/DIRECT-166212).
     */
    override fun execute(inputData: UpdateGeoInputData, prevState: YtState?): YtState? {

        if (prevState == null) {
            logger.info("Started processing")
            return YtState().apply {
                nextRow = 0
                totalRowCount = getTableRowCount(inputData)
            }
        }

        val startRow = prevState.nextRow
        val totalRows = prevState.totalRowCount
        val endRow = min(startRow + CHUNK_SIZE, totalRows)
        if (startRow >= totalRows) {
            logger.info("Finished processing, processed in total $totalRows rows")
            return null
        }

        try {
            logger.info("New iteration, processing rows from $startRow to $endRow out of $totalRows rows")
            val updatedCount = executeInternal(inputData, startRow, endRow)
            logger.info("Updated $updatedCount rows from $startRow to $endRow")
        } catch (e: RuntimeException) {
            logger.error("Caught exception: ${e.message}")
            throw e
        }

        return YtState().apply {
            nextRow = endRow
            totalRowCount = totalRows
        }
    }

    private fun isPlusRegion(regionId: Long, geoTargeting: Set<Long>) = geoTargeting.contains(regionId)

    private fun isMinusRegion(regionId: Long, geoTargeting: Set<Long>) = geoTargeting.contains(-regionId)

    private fun isRegionSelected(geoTargeting: Set<Long>, regionId: Long, parentId: Long? = null): Boolean {
        val parId = parentId ?: geoTree.getRegion(regionId).parent.id
        if (isMinusRegion(regionId, geoTargeting) || parId == regionId) {
            return false
        }
        return isPlusRegion(regionId, geoTargeting) || isRegionSelected(geoTargeting, parId)
    }

    fun ytOperator(inputData: YtInputData): YtOperator {
        val ytCluster = YtCluster.parse(inputData.ytCluster)
        return ytProvider.getOperator(ytCluster)
    }

    fun ytTable(inputData: YtInputData): YtTable {
        return YtTable(inputData.tablePath)
    }

    fun getTableRowCount(inputData: YtInputData): Long {
        return ytOperator(inputData).readTableRowCount(ytTable(inputData))
    }

    protected fun updatedGeoTargeting(
        inputData: UpdateGeoInputData,
        geoTargeting: Set<Long>
    ): Set<Long> {
        var updatedGeoTargeting = geoTargeting
        inputData.regionUpdates.forEach { updatedGeoTargeting = updatedGeoTargeting(it, updatedGeoTargeting) }
        return updatedGeoTargeting
    }

    private fun updatedGeoTargeting(
        inputData: UpdateRegionParam,
        geoTargeting: Set<Long>
    ): Set<Long> {
        val regionId = inputData.regionId
        val oldParentId = inputData.oldParentId
        val newParentId = inputData.newParentId
        val isNewParentSelected = isRegionSelected(geoTargeting, newParentId)
        if (isRegionSelected(geoTargeting, regionId, oldParentId)) {
            if (isPlusRegion(regionId, geoTargeting)) {
                if (isNewParentSelected) {
                    return geoTargeting.minus(regionId)
                }
            } else {
                if (!isNewParentSelected) {
                    return geoTargeting.plus(regionId)
                }
            }
        } else {
            if (isMinusRegion(regionId, geoTargeting)) {
                if (!isNewParentSelected) {
                    return geoTargeting.minus(-regionId)
                }
            } else {
                if (isNewParentSelected) {
                    return geoTargeting.plus(-regionId)
                }
            }
        }
        return geoTargeting
    }

    protected fun plusRegionsFromGeo(geo: Set<Long>): List<Long> {
        return geo.filter { it > 0 }
    }

    protected fun minusRegionsFromGeo(geo: Set<Long>): List<Long> {
        return geo.filter { it < 0 }.map { -it }
    }

    protected fun geoFromRegions(plusRegions: List<Long>?, minusRegions: List<Long>?): Set<Long> {
        return buildSet {
            plusRegions?.let { addAll(it) }
            minusRegions?.let { addAll(it.map { r -> -r }) }
        }
    }

    protected fun toLongSet(set: Set<Int>) = set.map { it.toLong() }.toSet()

    protected fun toIntSet(set: Set<Long>) = set.map { it.toInt() }.toSet()

    protected fun needResync(campaign: Campaign) = !campaign.statusArchived && campaign.orderId > 0

    protected fun getResyncPriority(campaign: Campaign) =
        if (campaign.statusShow) BS_RESYNC_PRIORITY_HIGH else BS_RESYNC_PRIORITY_LOW

    fun <T, R : ConvertibleTableRow<T>> readFromYtTable(
        inputData: YtInputData,
        tableRow: R,
        start: Long,
        end: Long
    ): List<T> {
        val ytTable = ytTable(inputData)
        val ytOperator = ytOperator(inputData)
        val result = mutableListOf<T>()
        ytOperator.readTableByRowRange(
            ytTable,
            { result.add(it.convert()) },
            tableRow,
            start,
            end
        )
        return result
    }

    abstract fun executeInternal(inputData: UpdateGeoInputData, start: Long, end: Long): Int
}
