package ru.yandex.direct.oneshot.oneshots.updateretconditionswithlal;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionOperationFactory;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingGoalsGoalType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.dbschema.ppcdict.tables.LalSegments.LAL_SEGMENTS;

@Component
@Approvers({"elwood"})
@Multilaunch
public class UpdateRetargetingConditionsWithLalSegmentsOneshot implements ShardedOneshot<InputData, Void> {
    private static final Logger logger = LoggerFactory.getLogger(UpdateRetargetingConditionsWithLalSegmentsOneshot.class);

    public static final int CHUNK_SIZE = 500;

    private final DslContextProvider dslContextProvider;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final RetargetingConditionOperationFactory retargetingConditionOperationFactory;

    @Autowired
    public UpdateRetargetingConditionsWithLalSegmentsOneshot(
            DslContextProvider dslContextProvider,
            RetargetingConditionRepository retargetingConditionRepository,
            RetargetingConditionOperationFactory retargetingConditionOperationFactory
    ) {
        this.dslContextProvider = dslContextProvider;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.retargetingConditionOperationFactory = retargetingConditionOperationFactory;
    }

    @Override
    public Void execute(InputData inputData, Void prevState, int shard) {
        Map<Long, List<Long>> retCondIdToGoalsMap = dslContextProvider.ppc(shard)
                .select(RETARGETING_GOALS.RET_COND_ID, RETARGETING_GOALS.GOAL_ID)
                .from(RETARGETING_GOALS)
                .where(RETARGETING_GOALS.GOAL_TYPE.eq(RetargetingGoalsGoalType.lal_segment))
                .fetchGroups(RETARGETING_GOALS.RET_COND_ID, RETARGETING_GOALS.GOAL_ID);

        logger.info("Found {} retargeting conditions with lal-segments", retCondIdToGoalsMap.size());

        List<Long> goalIds = retCondIdToGoalsMap
                .entrySet()
                .stream()
                .flatMap(entry -> entry.getValue().stream())
                .distinct()
                .sorted()
                .collect(Collectors.toList());

        logger.info("Found {} lal-segments used in retargeting conditions", goalIds.size());

        Map<Long, Long> lalToParentGoalMap = dslContextProvider.ppcdict()
                .select(LAL_SEGMENTS.LAL_SEGMENT_ID, LAL_SEGMENTS.PARENT_GOAL_ID)
                .from(LAL_SEGMENTS)
                .where(LAL_SEGMENTS.LAL_SEGMENT_ID.in(goalIds))
                .fetchMap(LAL_SEGMENTS.LAL_SEGMENT_ID, LAL_SEGMENTS.PARENT_GOAL_ID);

        List<Long> retCondIds = retCondIdToGoalsMap.keySet().stream().sorted().collect(Collectors.toList());

        for (var retCondChunk : Iterables.partition(retCondIds, CHUNK_SIZE)) {
            logger.info("Start iteration with chunk of {} retargeting conditions: {}",
                    retCondChunk.size(), retCondChunk);

            Map<Long, String> retCondIdToJsonMap = dslContextProvider.ppc(shard)
                    .select(RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_JSON)
                    .from(RETARGETING_CONDITIONS)
                    .where(RETARGETING_CONDITIONS.RET_COND_ID.in(retCondChunk))
                    .fetchMap(RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_JSON);

            List<AppliedChanges<RetargetingCondition>> conditionJsonChanges = retCondChunk.stream()
                    .map(retCondId -> {
                        String conditionJson = retCondIdToJsonMap.get(retCondId);
                        if (conditionJson == null) {
                            logger.warn("Can't find retargeting condition ret_cond_id={} in shard {}",
                                    retCondId, shard);
                            return null;
                        }

                        List<Rule> conditionRules = RetargetingConditionMappings.rulesFromJson(conditionJson);
                        conditionRules.forEach(rule -> rule.getGoals().forEach(goal -> {
                            if (goal.getUnionWithId() == null && lalToParentGoalMap.containsKey(goal.getId())) {
                                goal.setUnionWithId(lalToParentGoalMap.get(goal.getId()));
                            }
                        }));

                        List<Rule> oldConditionRules = RetargetingConditionMappings.rulesFromJson(conditionJson);
                        if (inputData.getForceResync() != null && inputData.getForceResync()) {
                            // заменяем уже ранее выставленный union_with_id на null,
                            // чтобы стригерить изменение в условии ретаргентинга
                            oldConditionRules.stream()
                                    .map(Rule::getGoals)
                                    .flatMap(Collection::stream)
                                    .filter(goal -> goal.getUnionWithId() != null)
                                    .findFirst()
                                    .ifPresent(goal -> goal.setUnionWithId(null));
                        }

                        return new ModelChanges<>(retCondId, RetargetingCondition.class)
                                .process(conditionRules, RetargetingCondition.RULES)
                                .applyTo((RetargetingCondition) new RetargetingCondition()
                                        .withId(retCondId)
                                        .withRules(oldConditionRules));
                    })
                    .filter(Objects::nonNull)
                    .filter(AppliedChanges::hasActuallyChangedProps)
                    .collect(Collectors.toList());

            logger.info("Going to update {} retargeting conditions: {}", conditionJsonChanges.size(),
                    conditionJsonChanges.stream().map(change -> change.getModel().getId()).collect(Collectors.toList()));

            retargetingConditionRepository.update(shard, conditionJsonChanges);

            logger.info("Update finished, start resyncing");
            retargetingConditionOperationFactory.retConditionPostUpdate(conditionJsonChanges, shard);

            logger.info("Resync finished");
        }
        return null;
    }

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        return ValidationResult.success(inputData);
    }
}
