package ru.yandex.direct.oneshot.worker;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.app.OneshotVersionControl;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.model.LaunchStatus;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceGuard;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.tracing.real.RealTrace;

public class ExecutionWorker implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(ExecutionWorker.class);

    // задачи, которые были поставлены в очередь на запуск executionWorker и не завершились
    private static final Set<OneshotExecuteTask<?, ?, ?>> tasksInProgress =
            Collections.synchronizedSet(new HashSet<>());

    private final OneshotRepository oneshotRepository;
    private final OneshotLaunchRepository launchRepository;
    private final OneshotLaunchDataRepository launchDataRepository;
    private final DslContextProvider dslContextProvider;
    private final OneshotVersionControl oneshotVersionControl;
    private final OneshotExecuteTaskFactory taskFactory;
    private final TraceHelper traceHelper;
    private final int idleTimeMillis;

    private final ThreadPoolExecutor executor;
    private final ArrayBlockingQueue<Runnable> queue;

    public ExecutionWorker(OneshotRepository oneshotRepository, OneshotLaunchRepository launchRepository,
                           OneshotLaunchDataRepository launchDataRepository, DslContextProvider dslContextProvider,
                           OneshotVersionControl oneshotVersionControl, OneshotExecuteTaskFactory taskFactory,
                           TraceHelper traceHelper, int idleTimeMillis) {
        this.oneshotRepository = oneshotRepository;
        this.launchRepository = launchRepository;
        this.launchDataRepository = launchDataRepository;
        this.dslContextProvider = dslContextProvider;
        this.oneshotVersionControl = oneshotVersionControl;
        this.taskFactory = taskFactory;
        this.traceHelper = traceHelper;
        this.idleTimeMillis = idleTimeMillis;

        this.queue = new ArrayBlockingQueue<>(1);
        this.executor = new ThreadPoolExecutor(32, 32, 1, TimeUnit.SECONDS, queue,
                new ThreadFactoryBuilder().setNameFormat("oneshot-executions-thread-%d").build());
    }

    public static void removeTaskInProgress(OneshotExecuteTask<?, ?, ?> task) {
        tasksInProgress.remove(task);
    }

    public static Set<OneshotExecuteTask<?, ?, ?>> getTasksInProgressCopy() {
        Set<OneshotExecuteTask<?, ?, ?>> copy;
        synchronized (tasksInProgress) {
            copy = new HashSet<>(tasksInProgress);
        }
        return copy;
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                Thread.sleep(idleTimeMillis);

                if (!oneshotVersionControl.isRunningCurrentVersion()) {
                    continue;
                }

                dslContextProvider.ppcdictTransaction(conf -> executeLaunch(conf.dsl()));

            } catch (RuntimeException e) {
                logger.error("unexpected error in execution loop", e);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        logger.warn("thread was interrupted");
        shutdownExecutionTasks();
    }

    private void executeLaunch(DSLContext transactionContext) {
        if (queue.remainingCapacity() == 0) {
            return;
        }

        OneshotLaunchData launchData = launchDataRepository.getReadyLaunchDataForUpdate(transactionContext);
        if (launchData == null) {
            return;
        }

        OneshotLaunch launch = launchRepository.getSafe(launchData.getLaunchId());
        Oneshot oneshot = oneshotRepository.getSafe(launch.getOneshotId());

        updateLaunchDataForStart(transactionContext, launchData);

        Trace trace = RealTrace.builder()
                .withTraceId(launch.getTraceId())
                .withSpanId(0L)
                .withInfo(traceHelper.getService(), "oneshotExecution", "")
                .build();

        OneshotExecuteTask<?, ?, ?> task;
        try (TraceGuard g = traceHelper.guard(trace)) {
            logger.info("oneshot launch was locked for execution: class = {}, shard = {}",
                    oneshot.getClassName(), launchData.getShard());

            task = taskFactory.createOneshotExecuteTask(oneshot, launch, launchData);
            if (task == null) { // ваншот отсутствует в коде
                if (oneshotVersionControl.isRunningCurrentVersion()) {
                    logger.error("Actual version of code doesn't contain oneshot ({})", oneshot.getClassName());
                    updateStatusAndFinishTime(transactionContext, launchData, LaunchStatus.FAILED);
                } else {
                    logger.warn("This version of code doesn't contain this oneshot yet ({})", oneshot.getClassName());
                    updateStatusAndFinishTime(transactionContext, launchData, LaunchStatus.READY);
                }
                return;
            }

            logger.info("oneshot launch task is being passed to executor: class = {}, shard = {}",
                    oneshot.getClassName(), launchData.getShard());
        }

        tasksInProgress.add(task);
        executor.execute(new OneshotExecuteTaskMonitor(task));
    }

    private void updateStatusAndFinishTime(DSLContext transactionContext,
                                           OneshotLaunchData launchData, LaunchStatus launchStatus) {
        launchData.withLaunchStatus(launchStatus)
                .withFinishTime(LocalDateTime.now());
        launchDataRepository.updateStatusAndFinishTime(transactionContext, launchData);
    }

    @VisibleForTesting
    void updateLaunchDataForStart(DSLContext transactionContext,
                                  OneshotLaunchData launchData) {
        launchData.withLaunchStatus(LaunchStatus.IN_PROGRESS);
        launchData.setLastActiveTime(LocalDateTime.now());
        if (launchData.getLaunchTime() == null) {
            launchData.setLaunchTime(LocalDateTime.now());
        }

        Set<String> revisions = launchData.getLaunchedRevisions() != null ?
                new HashSet<>(launchData.getLaunchedRevisions()) : new HashSet<>();
        revisions.add(oneshotVersionControl.getCurrentVersion());
        launchData.withLaunchedRevisions(revisions);

        launchDataRepository.updateLaunchDataForStart(transactionContext, launchData);
    }

    private void shutdownExecutionTasks() {
        logger.warn("initializing shutdown of active execution tasks");

        OneshotExecuteTaskMonitor.shutdownTasks();
        executor.shutdown();

        try {
            logger.warn("waiting for execution tasks termination");
            if (executor.awaitTermination(60, TimeUnit.MINUTES)) {
                logger.warn("execution tasks are successfully completed and terminated");
            } else {
                logger.error("timeout for termination has expired");
            }
        } catch (InterruptedException e) {
            logger.error("thread was unexpectedly interrupted, ", e);
        }
    }
}
