package ru.yandex.direct.oneshot.worker;

import java.time.Duration;
import java.time.LocalDateTime;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class OneshotAliveChecker implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(OneshotAliveChecker.class);

    private static final Duration IDLE_TIME = Duration.ofSeconds(5);
    private static final Duration NOT_ACTIVE_MAX_DELAY = Duration.ofMinutes(30);

    private final DslContextProvider dslContextProvider;
    private final OneshotLaunchDataRepository launchDataRepository;

    @Autowired
    public OneshotAliveChecker(DslContextProvider dslContextProvider,
                               OneshotLaunchDataRepository launchDataRepository) {
        this.dslContextProvider = dslContextProvider;
        this.launchDataRepository = launchDataRepository;
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                updateAliveOneshotsLastActiveTime();
                forcePauseDeadOneshots();
                try {
                    Thread.sleep(IDLE_TIME.toMillis());
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            } catch (Throwable e) {
                logger.error("Unexpected exception, thread will continue", e);
            }
        }
        logger.info("thread was interrupted");
    }

    private void updateAliveOneshotsLastActiveTime() {
        var activeTasksLaunchDatas = StreamEx.of(ExecutionWorker.getTasksInProgressCopy())
                .map(OneshotExecuteTask::getLaunchData)
                .toList();

        if (!activeTasksLaunchDatas.isEmpty()) {
            logger.info("Updating last_active_time for {} alive oneshots with launch data ids: {}",
                    activeTasksLaunchDatas.size(),
                    mapList(activeTasksLaunchDatas, OneshotLaunchData::getId));
        }

        dslContextProvider.ppcdictTransaction(conf ->
                launchDataRepository.updateLastActiveTime(conf.dsl(), activeTasksLaunchDatas, LocalDateTime.now()));
    }

    private void forcePauseDeadOneshots() {
        dslContextProvider.ppcdictTransaction(conf -> {
            var timeThreshold = LocalDateTime.now().minus(NOT_ACTIVE_MAX_DELAY);
            var deadLaunchDatas = launchDataRepository.selectNotActiveLaunchDatasForUpdate(
                    conf.dsl(), timeThreshold);

            if (!deadLaunchDatas.isEmpty()) {
                logger.info("cancelling {} not active launch datas since {} with ids: {}",
                        deadLaunchDatas.size(),
                        timeThreshold,
                        mapList(deadLaunchDatas, OneshotLaunchData::getId));

                launchDataRepository.pauseLaunchDatas(conf.dsl(), deadLaunchDatas);
            }
        });
    }
}
