package ru.yandex.direct.oneshot.worker;

import java.time.LocalDateTime;
import java.util.Map;
import java.util.Optional;

import com.google.common.base.Stopwatch;
import com.google.gson.Gson;
import org.apache.commons.lang3.StringUtils;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.model.LaunchStatus;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;
import ru.yandex.direct.oneshot.util.GsonUtils;
import ru.yandex.direct.oneshot.worker.def.BaseOneshot;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.utils.ThreadUtils;

public class OneshotExecuteTask<T extends BaseOneshot<V>, V, S> implements Runnable {

    private static final Logger logger = LoggerFactory.getLogger(OneshotExecuteTask.class);
    private static final Gson gson = GsonUtils.getGSON();

    private final TraceHelper traceHelper;
    private final PpcProperty<Map<String, Integer>> relaxedPercentsProp;
    private final DslContextProvider dslContextProvider;
    private final OneshotLaunchDataRepository launchDataRepository;

    private final Oneshot oneshot;
    private final T oneshotBean;
    private final OneshotLaunch launch;
    private final S stateData;

    private final OneshotLaunchData launchData;
    private final OneshotExecutionStrategy<T, S> executionStrategy;

    private volatile boolean shutdown = false;

    public OneshotExecuteTask(TraceHelper traceHelper,
                              PpcProperty<Map<String, Integer>> relaxedPercentsProp,
                              DslContextProvider dslContextProvider,
                              OneshotLaunchDataRepository launchDataRepository,
                              Oneshot oneshot,
                              T oneshotBean,
                              OneshotLaunch launch,
                              S stateData,
                              OneshotLaunchData launchData,
                              OneshotExecutionStrategy executionStrategy) {
        this.traceHelper = traceHelper;
        this.relaxedPercentsProp = relaxedPercentsProp;
        this.dslContextProvider = dslContextProvider;
        this.launchDataRepository = launchDataRepository;
        this.oneshot = oneshot;
        this.oneshotBean = oneshotBean;
        this.launch = launch;
        this.stateData = stateData;
        this.launchData = launchData;
        this.executionStrategy = executionStrategy;
    }

    @Override
    public void run() {
        var simpleClassName = StringUtils.substringAfterLast(oneshot.getClassName(), ".");
        try (var g = traceHelper.guard("oneshotExecute." + simpleClassName, "",
                launch.getTraceId(), launchData.getSpanId())
        ) {
            try {
                S stateData = this.stateData;
                boolean needExit = false;
                do {
                    logger.info("starting iteration...");
                    var stopwatch = Stopwatch.createStarted();
                    final S localStateData = executionStrategy.apply(oneshotBean, stateData);
                    logger.info("iteration finished, saving state...");

                    stateData = localStateData;
                    needExit = dslContextProvider.ppcdictTransactionResult(conf -> {
                        DSLContext dslContext = conf.dsl();

                        updateState(dslContext, localStateData);
                        if (localStateData == null) {
                            logger.info("oneshot returned null state, finished!");
                            updateStatusAndFinishTime(dslContext, LaunchStatus.DONE);
                            return true;
                        }

                        OneshotLaunchData selected = launchDataRepository.selectForUpdate(dslContext,
                                launchData.getId());
                        LaunchStatus launchStatus = selected.getLaunchStatus();

                        if (launchStatus == LaunchStatus.PAUSE_REQUESTED || launchStatus == LaunchStatus.PAUSED ||
                                launchStatus == LaunchStatus.CANCELED || shutdown) {
                            if (launchStatus == LaunchStatus.PAUSE_REQUESTED) {
                                logger.info("oneshot is in {} status, change status to {}, break execution loop",
                                        LaunchStatus.PAUSE_REQUESTED, LaunchStatus.PAUSED);
                                launchData.withLaunchStatus(LaunchStatus.PAUSED);
                                launchDataRepository.updateStatus(dslContext, launchData);
                            } else if (shutdown) {
                                logger.info("oneshot app is shutting down, stop execution and change status to {}",
                                        LaunchStatus.READY);
                                launchData.withLaunchStatus(LaunchStatus.READY);
                                launchDataRepository.updateStatus(dslContext, launchData);
                            } else {
                                logger.info("oneshot is in {} status, break execution loop",
                                        launchData.getLaunchStatus());
                            }
                            updateFinishTime(dslContext);
                            return true;
                        }
                        return false;
                    });
                    if (!needExit) {
                        int relaxedPercent = Optional.ofNullable(relaxedPercentsProp.get())
                                .filter(m -> m.containsKey(simpleClassName))
                                .map(m -> m.get(simpleClassName))
                                .orElse(0);
                        if (relaxedPercent > 0) {
                            try (var p = Trace.current().profile("relaxed:sleep")) {
                                var sleepDuration = stopwatch.elapsed().multipliedBy(relaxedPercent).dividedBy(100);
                                ThreadUtils.sleep(sleepDuration);
                            }
                        }
                    }
                } while (stateData != null && !needExit);

            } catch (RuntimeException e) {
                String clsName = oneshotBean.getClass().getCanonicalName();
                logger.error("error while executing oneshot " + clsName, e);
                if (oneshot.getPausedStatusOnFail()) {
                    tryUpdateStatusAndFinishTime(LaunchStatus.PAUSED);
                } else {
                    tryUpdateStatusAndFinishTime(LaunchStatus.FAILED);
                }
            }
        }
    }

    private void updateState(DSLContext transactionContext, S state) {
        String stateStr = state != null ? gson.toJson(state) : null;
        launchData.withState(stateStr);
        launchDataRepository.updateState(transactionContext, launchData);
    }

    private void tryUpdateStatusAndFinishTime(LaunchStatus launchStatus) {
        try {
            updateStatusAndFinishTime(dslContextProvider.ppcdict(), launchStatus);
        } catch (RuntimeException e) {
            logger.error("error while trying to set oneshot launch data status to " + launchStatus, e);
        }
    }

    private void updateStatusAndFinishTime(DSLContext transactionContext, LaunchStatus launchStatus) {
        launchData.withLaunchStatus(launchStatus)
                .withFinishTime(LocalDateTime.now());
        launchDataRepository.updateStatusAndFinishTime(transactionContext, launchData);
    }

    private void updateFinishTime(DSLContext transactionContext) {
        launchData.withFinishTime(LocalDateTime.now());
        launchDataRepository.updateFinishTime(transactionContext, launchData);
    }

    public void setShutdown(boolean shutdown) {
        this.shutdown = shutdown;
    }

    public OneshotLaunchData getLaunchData() {
        return launchData;
    }
}
