package ru.yandex.direct.oneshot.worker;

import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import one.util.streamex.StreamEx;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;
import ru.yandex.direct.oneshot.util.GsonUtils;
import ru.yandex.direct.oneshot.worker.def.BaseOneshot;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.tracing.TraceHelper;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectOneshotInputDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectShardedOneshotStateDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectSimpleOneshotStateDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.findOneshotBeanByClassName;

@Service
public class OneshotExecuteTaskFactory {

    private static final Gson gson = GsonUtils.getGSON();

    private final ApplicationContext appContext;
    private final DslContextProvider dslContextProvider;
    private final OneshotLaunchDataRepository launchDataRepository;
    private final TraceHelper traceHelper;
    private final PpcProperty<Map<String, Integer>> relaxedPercentsProp;

    public OneshotExecuteTaskFactory(ApplicationContext appContext,
                                     DslContextProvider dslContextProvider,
                                     OneshotLaunchDataRepository launchDataRepository,
                                     TraceHelper traceHelper,
                                     PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.appContext = appContext;
        this.dslContextProvider = dslContextProvider;
        this.launchDataRepository = launchDataRepository;
        this.traceHelper = traceHelper;
        this.relaxedPercentsProp = ppcPropertiesSupport.get(PpcPropertyNames.ONESHOT_RELAXED_PAUSE_PERCENT);
    }

    @Nullable
    public OneshotExecuteTask<?, ?, ?> createOneshotExecuteTask(Oneshot oneshot,
                                                                OneshotLaunch launch,
                                                                OneshotLaunchData launchData) {
        BaseOneshot oneshotBean = findOneshotBeanByClassName(appContext, oneshot.getClassName());
        if (oneshotBean == null) {
            return null;
        }

        OneshotType oneshotType = OneshotType.getOneshotType(oneshotBean);
        return oneshotType.createTask(traceHelper, relaxedPercentsProp, dslContextProvider, launchDataRepository,
                oneshotBean, oneshot, launch, launchData);
    }

    @SuppressWarnings("UnstableApiUsage")
    private enum OneshotType {

        SIMPLE(SimpleOneshot.class) {
            @Override
            public OneshotExecuteTask<?, ?, ?> createTask(TraceHelper traceHelper,
                                                          PpcProperty<Map<String, Integer>> relaxedPercentsProp,
                                                          DslContextProvider dslContextProvider,
                                                          OneshotLaunchDataRepository launchDataRepository,
                                                          BaseOneshot oneshotBean, Oneshot oneshot,
                                                          OneshotLaunch launch, OneshotLaunchData launchData) {
                assertOneshotConsistent(oneshotBean, oneshot, launchData);
                return createTaskInternal(traceHelper, relaxedPercentsProp, dslContextProvider,
                        launchDataRepository, oneshot,
                        (SimpleOneshot<?, ?>) oneshotBean, launch, launchData);
            }

            public void assertOneshotConsistent(BaseOneshot oneshotBean, Oneshot oneshot,
                                                OneshotLaunchData launchData) {
                String clsName = oneshotBean.getClass().getCanonicalName();
                checkState(oneshotBean instanceof SimpleOneshot,
                        "Oneshot bean must be instance of SimpleOneshot, but it's not (%s)", clsName);
                checkState(!oneshot.getSharded(),
                        "Oneshot is sharded in the database but not in the code (%s)", clsName);
                checkState(launchData.getShard() == null,
                        "Oneshot is not sharded in the code but has shard in launch data table (%s)", clsName);
            }

            private <V, S> OneshotExecuteTask<SimpleOneshot<V, S>, V, S> createTaskInternal(
                    TraceHelper traceHelper,
                    PpcProperty<Map<String, Integer>> relaxedPercentsProp,
                    DslContextProvider dslContextProvider,
                    OneshotLaunchDataRepository launchDataRepository,
                    Oneshot oneshot,
                    SimpleOneshot<V, S> simpleOneshotBean, OneshotLaunch launch,
                    OneshotLaunchData launchData) {

                TypeToken<V> inputDataType = detectOneshotInputDataType(simpleOneshotBean);
                V inputData = gson.fromJson(launch.getParams(), inputDataType.getType());

                TypeToken<S> stateDataType = detectSimpleOneshotStateDataType(simpleOneshotBean);
                S stateData = gson.fromJson(launchData.getState(), stateDataType.getType());

                OneshotExecutionStrategy<SimpleOneshot<V, S>, S> executionStrategy =
                        new OneshotRetryableExecutionStrategy<>(
                                (o, s) -> o.execute(inputData, s),
                                oneshot.getRetries(), oneshot.getRetryTimeoutSeconds()
                        );
                return new OneshotExecuteTask<>(traceHelper, relaxedPercentsProp, dslContextProvider,
                        launchDataRepository,
                        oneshot, simpleOneshotBean, launch, stateData, launchData, executionStrategy);
            }
        },

        SHARDED(ShardedOneshot.class) {
            @Override
            public OneshotExecuteTask<?, ?, ?> createTask(TraceHelper traceHelper,
                                                          PpcProperty<Map<String, Integer>> relaxedPercentsProp,
                                                          DslContextProvider dslContextProvider,
                                                          OneshotLaunchDataRepository launchDataRepository,
                                                          BaseOneshot oneshotBean, Oneshot oneshot,
                                                          OneshotLaunch launch, OneshotLaunchData launchData) {
                assertOneshotConsistent(oneshotBean, oneshot, launchData);
                return createTaskInternal(traceHelper, relaxedPercentsProp, dslContextProvider, launchDataRepository,
                        oneshot, (ShardedOneshot<?, ?>) oneshotBean, launch, launchData);
            }

            private void assertOneshotConsistent(BaseOneshot oneshotBean, Oneshot oneshot,
                                                 OneshotLaunchData launchData) {
                String clsName = oneshotBean.getClass().getCanonicalName();
                checkState(oneshotBean instanceof ShardedOneshot,
                        "Oneshot bean must be instance of ShardedOneshot, but it's not (%s)", clsName);
                checkState(oneshot.getSharded(),
                        "Oneshot is sharded in the code but not in the database (%s)", clsName);
                checkState(launchData.getShard() != null,
                        "Oneshot is sharded in the code but has no shard in launch data table (%s)", clsName);
            }

            private <V, S> OneshotExecuteTask<ShardedOneshot<V, S>, V, S> createTaskInternal(
                    TraceHelper traceHelper,
                    PpcProperty<Map<String, Integer>> relaxedPercentsProp,
                    DslContextProvider dslContextProvider,
                    OneshotLaunchDataRepository launchDataRepository,
                    Oneshot oneshot,
                    ShardedOneshot<V, S> shardedOneshotBean, OneshotLaunch launch,
                    OneshotLaunchData launchData) {

                TypeToken<V> inputDataType = detectOneshotInputDataType(shardedOneshotBean);
                V inputData = gson.fromJson(launch.getParams(), inputDataType.getType());

                TypeToken<S> stateDataType = detectShardedOneshotStateDataType(shardedOneshotBean);
                S stateData = gson.fromJson(launchData.getState(), stateDataType.getType());

                OneshotExecutionStrategy<ShardedOneshot<V, S>, S> executionStrategy =
                        new OneshotRetryableExecutionStrategy<>(
                                (o, s) -> o.execute(inputData, s, launchData.getShard()),
                                oneshot.getRetries(), oneshot.getRetryTimeoutSeconds()
                        );
                return new OneshotExecuteTask<>(traceHelper, relaxedPercentsProp, dslContextProvider,
                        launchDataRepository,
                        oneshot, shardedOneshotBean, launch, stateData, launchData, executionStrategy);
            }
        };

        protected abstract OneshotExecuteTask<?, ?, ?> createTask(
                TraceHelper traceHelper,
                PpcProperty<Map<String, Integer>> relaxedPercentsProp,
                DslContextProvider dslContextProvider,
                OneshotLaunchDataRepository launchDataRepository,
                BaseOneshot oneshotBean, Oneshot oneshot,
                OneshotLaunch launch, OneshotLaunchData launchData);

        private static ImmutableMap<Class<?>, OneshotType> lookup;

        static {
            ImmutableMap.Builder<Class<?>, OneshotType> builder = ImmutableMap.builder();
            Stream.of(OneshotType.values()).forEach(oneshotType ->
                    builder.put(oneshotType.baseOneshotClass, oneshotType));
            lookup = builder.build();
        }

        public static OneshotType getOneshotType(BaseOneshot oneshotBean) {
            Class<?> key = StreamEx.of(lookup.keySet())
                    .findFirst(clazz -> clazz.isAssignableFrom(oneshotBean.getClass()))
                    .orElseThrow(() -> new IllegalStateException(
                            String.format("oneshot bean doesn't implement any supported interface: %s",
                                    oneshotBean.getClass())));
            return lookup.get(key);
        }

        private Class<?> baseOneshotClass;

        OneshotType(Class<?> baseOneshotClass) {
            this.baseOneshotClass = baseOneshotClass;
        }
    }
}
