package ru.yandex.direct.oneshot.worker;

import java.time.Duration;
import java.util.function.BiFunction;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.oneshot.worker.def.BaseOneshot;
import ru.yandex.direct.utils.ThreadUtils;

public class OneshotRetryableExecutionStrategy<T extends BaseOneshot<?>, S> implements OneshotExecutionStrategy<T, S> {
    private static final Logger logger = LoggerFactory.getLogger(OneshotRetryableExecutionStrategy.class);

    private final BiFunction<T, S, S> runner;
    private final int retries;
    private final int retryTimeout;

    public OneshotRetryableExecutionStrategy(BiFunction<T, S, S> runner, int retries, int retryTimeout) {
        this.runner = runner;
        this.retries = retries;
        this.retryTimeout = retryTimeout;
    }

    @Override
    public S apply(T oneshot, S state) {
        int currentRetries = retries;

        while (currentRetries >= 0) {
            try {
                return runner.apply(oneshot, state);
            } catch (RuntimeException e) {
                if (currentRetries == 0) {
                    throw new AllOneshotRetriesFailedException("All retries fails", e);
                } else {
                    var msg = String.format(
                            "Oneshot %s iteration failed, retrying.. Number of retries left %d",
                            oneshot.getClass(), currentRetries
                    );
                    logger.error(msg, e);
                }
                currentRetries--;
                if (retryTimeout > 0) {
                    ThreadUtils.sleep(Duration.ofSeconds(retryTimeout));
                }
            }
        }
        return null;
    }

    /**
     * Случается когда израсходовано количество попыток (Oneshot.retries) выполнить итерацию
     */
    public static class AllOneshotRetriesFailedException extends RuntimeException {
        AllOneshotRetriesFailedException(String message, RuntimeException cause) {
            super(message, cause);
        }
    }
}
