package ru.yandex.direct.oneshot.worker;

import com.google.common.reflect.TypeToken;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.app.OneshotVersionControl;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchValidationStatus;
import ru.yandex.direct.oneshot.util.GsonUtils;
import ru.yandex.direct.oneshot.worker.def.BaseOneshot;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceGuard;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.oneshot.worker.WorkerUtils.detectOneshotInputDataType;
import static ru.yandex.direct.oneshot.worker.WorkerUtils.findOneshotBeanByClassName;

@SuppressWarnings("UnstableApiUsage")
public class ValidationWorker implements Runnable {

    private static final Logger logger = LoggerFactory.getLogger(ValidationWorker.class);

    private final ApplicationContext appContext;
    private final OneshotRepository oneshotRepository;
    private final OneshotLaunchRepository launchRepository;
    private final DslContextProvider dslContextProvider;
    private final OneshotVersionControl oneshotVersionControl;
    private final TraceHelper traceHelper;
    private final int idleTimeMillis;

    public ValidationWorker(ApplicationContext appContext, OneshotRepository oneshotRepository,
                            OneshotLaunchRepository launchRepository,
                            DslContextProvider dslContextProvider,
                            OneshotVersionControl oneshotVersionControl, TraceHelper traceHelper,
                            int idleTimeMillis) {
        this.appContext = appContext;
        this.oneshotRepository = oneshotRepository;
        this.launchRepository = launchRepository;
        this.dslContextProvider = dslContextProvider;
        this.oneshotVersionControl = oneshotVersionControl;
        this.traceHelper = traceHelper;
        this.idleTimeMillis = idleTimeMillis;
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                Thread.sleep(idleTimeMillis);

                if (!oneshotVersionControl.isRunningCurrentVersion()) {
                    continue;
                }

                dslContextProvider.ppcdictTransaction(conf -> validateLaunch(conf.dsl()));

            } catch (RuntimeException e) {
                logger.error("unexpected error in validation loop, continue", e);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        logger.warn("thread was interrupted");
    }

    private void validateLaunch(DSLContext transactionContext) {
        OneshotLaunch readyLaunch = launchRepository.getReadyLaunchForUpdate(transactionContext);
        if (readyLaunch == null) {
            return;
        }

        try (TraceGuard g = traceHelper.guard("oneshotValidation", "", readyLaunch.getTraceId(), 0L)) {
            Oneshot oneshot = oneshotRepository.getSafe(readyLaunch.getOneshotId());

            boolean validationSuccessful = validateOneshot(appContext, oneshot.getClassName(), readyLaunch.getParams());

            OneshotLaunchValidationStatus status = validationSuccessful ?
                    OneshotLaunchValidationStatus.VALID :
                    OneshotLaunchValidationStatus.INVALID;
            launchRepository.updateValidationStatus(transactionContext, readyLaunch.getId(), status);
        }
    }

    public static <V, T extends BaseOneshot<V>> boolean validateOneshot(ApplicationContext appContext,
                                                                        String oneshotClassName,
                                                                        String params) {
        boolean validationSuccessful = false;
        try {
            @SuppressWarnings("unchecked")
            T oneshotBean = (T) findOneshotBeanByClassName(appContext, oneshotClassName);
            TypeToken<V> inputDataType = detectOneshotInputDataType(oneshotBean);

            V v = GsonUtils.getGSON().fromJson(params, inputDataType.getType());
            try (TraceProfile p = Trace.current().profile("validation")) {
                ValidationResult<V, Defect> vr = oneshotBean.validate(v);
                validationSuccessful = !vr.hasAnyErrors();
                logger.info("validation errors: {}", vr.flattenErrors());
                logger.info("validation warnings: {}", vr.flattenWarnings());
            }
        } catch (RuntimeException e) {
            logger.error("validation is failed", e);
        }
        return validationSuccessful;
    }
}
