#!/usr/bin/env python
# coding=utf-8
"""
mysql-метрики
"""

from datetime import datetime as dt
from subprocess import check_output, STDOUT, CalledProcessError

# после загрузки агент не перечитывает класс, но можно выгрузить его через applicable и загрузить обратно
class MysqlPullModule:
    def __init__(self, logger, registry, **params):
        self._logger = logger
        self._logprefix = self.__class__.__name__
        self.my_log('new instance with params={}'.format(params))

    def my_log(self, msg):
        self._logger.info('%s: %s' % (self._logprefix, msg))

    def my_check_output(self, cmd):
        try:
            out = check_output(cmd, stderr=STDOUT)
        except CalledProcessError as e:
            self.my_log('cmd output: ' + e.output)
            raise
        return out

    def put_mysql_status(self, instance, ts, consumer):
        self.my_log('get mysql status for instance ' + instance)

        # пока у нас нет локальной авторизации, юзер monitor может зайти - тут это плюс =(
        cmd = ['/usr/bin/mysql', '-u', 'root', '-h', 'localhost', '-S', '/var/run/mysqld.' + instance + '/mysqld.sock', '--batch', '-e' ,'show status']
        out = self.my_check_output(cmd)

        consumed_sensors = 0
        for line in out.split('\n'):
            try:
                name, value = line.strip().split()
                value = float(value)
            except:
                continue # ошибки парсинга - это обычно норма, просто игнорим

            consumer.gauge({'sensor': 'mysql_status.' + name, 'instance': instance}, ts, value)
            consumed_sensors += 1

        self.my_log('parsed mysql status for %s, %d sensors consumed' % (instance, consumed_sensors))

    def put_mysql_variables(self, instance, ts, consumer):
        self.my_log('get mysql variables for instance ' + instance)

        # пока у нас нет локальной авторизации, юзер monitor может зайти - тут это плюс =(
        cmd = ['/usr/bin/mysql', '-u', 'root', '-h', 'localhost', '-S', '/var/run/mysqld.' + instance + '/mysqld.sock', '--batch', '-e' ,'show variables']
        out = self.my_check_output(cmd)

        consumed_sensors = 0
        for line in out.split('\n'):
            try:
                name, value = line.strip().split()
                value = float(value)
            except:
                continue # ошибки парсинга - это обычно норма, просто игнорим

            consumer.gauge({'sensor': 'mysql_variables.' + name, 'instance': instance}, ts, value)
            consumed_sensors += 1

        self.my_log('parsed mysql variables for %s, %d sensors consumed' % (instance, consumed_sensors))

    def pull(self, ts, consumer):
        self.my_log('start pull method with utc timestamp %s' % (dt.utcfromtimestamp(ts / 1000.0).strftime('%Y-%m-%d %H:%M:%S')))

        for instance in self.my_check_output(['/usr/local/bin/lm', '--complete']).split('\n'):
            if not instance:
                continue
            try:
                self.put_mysql_status(instance, ts, consumer)
                self.put_mysql_variables(instance, ts, consumer)
            except Exception as e:
                self.my_log('cannot get mysql stats from %s: %s %s' % (instance, type(e), e))

        self.my_log('end pull method')


def applicable():
    try:
        out = check_output(['/usr/local/bin/lm', '--complete'], stderr=STDOUT)
        return any(x.strip() for x in out.split('\n')) # если есть хотя бы один установленный mysql - работаем
    except:
        return False


# эта часть выполняется только при генерации конфигов (запускается системным python)
# и никак не связана со сбором метрик (запускается встроенным solomon-agent python)
if __name__ == '__main__':
    import sys
    import json

    sys.path.extend(['share/dt_solomon', '/usr/local/share/dt_solomon'])
    from dts_common import service_config, python2_pull_config, system_pull_config

    # пути к файлу и директории - параметры подставит генератор конфига
    file_path = sys.argv[1]
    module_pkg = sys.argv[2]
    # имя сервиса генерируется из имени файла, для my_module.py будет my-module
    service = sys.argv[3]

    if not applicable():
        print '{}'
        sys.exit(0)

    config = [
        # "контейнер" для однотипных сенсоров https://wiki.yandex-team.ru/solomon/userguide/datamodel/
        # solomon будет забирать сенсоры по hostname:.../...?project=...&service=...
        service_config(
            service=service,
            project='direct',
            pull_interval='15s',
            modules=[
                python2_pull_config(file_path, module_pkg, "MysqlPullModule", params={}) # тут можно передать специфичные параметры, например, какие метрики нужны на данном хосте
            ],
            labels=[] # общие метки для всех модулей, почти никогда не нужны
        ),
    ]

    print json.dumps(config)
