#!/usr/bin/env python
# coding=utf-8
"""
zookeeper-метрики
"""

import os
import StringIO
import socket
from datetime import datetime as dt
from subprocess import check_output, STDOUT, CalledProcessError
from ConfigParser import ConfigParser


# после загрузки агент не перечитывает класс, но можно выгрузить его через applicable и загрузить обратно
class ZookeeperPullModule:
    default_path = "/var/lib/zookeeper"
    config2instance = {
        '/etc/zookeeper/conf/zoo.cfg': 'default',
        '/etc/zookeeper/ppc/zoo.cfg': 'ppc'
    }

    def __init__(self, logger, registry, **params):
        self._logger = logger
        self._logprefix = self.__class__.__name__
        self.my_log('new instance with params={}'.format(params))

    def my_log(self, msg):
        self._logger.info('%s: %s' % (self._logprefix, msg))

    def my_check_output(self, cmd):
        try:
            out = check_output(cmd, stderr=STDOUT)
        except CalledProcessError as e:
            self.my_log('cmd output: ' + e.output)
            raise
        return out


    def get_config(self, cmd_line):
        """
        определяем с каким конфигом запущен сервер зукипера
        """
        for config in self.config2instance:
            if config in cmd_line:
                return config

        return ''

   
    def read_config(self, config):
        """
        cчитываем конфиг зукипера
        """
        data = '[main]\n'
        zconfig = ConfigParser()
        with open(config) as fd:
            buf = StringIO.StringIO(data + fd.read())
        zconfig.readfp(buf)
        return zconfig


    def get_gc_stats(self, pid):
        """
        собираем данные из garbage collector'а
        """
        try:
            gc_stats = self.my_check_output(["sudo", "/usr/bin/jstat", "-gc", pid])
            (stat_names, values, _) = gc_stats.split('\n')
            stat_names = ['gc_' + i.lower() for i in stat_names.strip().split()]
            values = [int(float(i)) for i in values.strip().split()]
            return dict(zip(stat_names, values))

        except:
            self.my_log("can't get data from garbage collector (pid=%s)" % pid)
            return {}


    def get_dir_size(self, zk_config, pid):
        """
        получаем размер данных
        """
        try:
            snap_path = zk_config.get('main', 'dataDir') if zk_config.has_option('main', 'dataDir') else self.default_path
            log_path = zk_config.get('main', 'dataLogDir') if zk_config.has_option('main', 'dataLogDir') else self.default_path
            snap_path += '/version-2'
            log_path += '/version-2'
            list_snaps = [os.path.join(snap_path, i) for i in os.listdir(snap_path) if i.startswith('snapshot')]
            list_logs = [os.path.join(log_path, i) for i in os.listdir(log_path) if i.startswith('log')]
            size_snap = max((os.stat(name).st_mtime, os.stat(name).st_size) for name in list_snaps)[1]
            size_log = max((os.stat(name).st_mtime, os.stat(name).st_size) for name in list_logs)[1]
            return {'snap_size': size_snap, 'log_size': size_log}

        except Exception as e:
            self.my_log("can't get dirs sizes (pid=%s), %s %s" % (pid, type(e), e))
            return {}


    def get_mntr_stats(self, zk_config, pid):
        """
        получаем данные из вывода команды mntr, посланной зукипер-серверу
        """
        try:
            result = {}
            if zk_config.has_option('main', 'clientPort'):
                clientPort = zk_config.get('main', 'clientPort')
            else:
                self.my_log("can't find clientPort in zookeeper config")
                return result
    
            sock = socket.create_connection(('localhost', clientPort))
            sock.send('mntr')
            mntr_data = ''.join([i for i in sock.recv(8192)])
            sock.close()
    
            result = dict(i.split('\t') for i in mntr_data.split('\n') if i)
            if 'zk_server_state' in result:
                known_states = {'leader': 1, 'follower': 2, 'looking': 3}
                result['zk_server_state'] = known_states[result['zk_server_state']]
    
            return result

        except Exception as e:
            self.my_log("can't get stats from mntr (pid=%s), %s %s" % (pid, type(e), e))
            return {}


    def get_zk_status(self, pid, config):
        """
        получаем метрики для отправки в соломон в виде словаря
        (metric -> value)
        """
        results = {}

        results.update(self.get_gc_stats(pid))
        zk_config = self.read_config(config)
        results.update(self.get_dir_size(zk_config, pid))
        results.update(self.get_mntr_stats(zk_config, pid))

        return results


    def pull(self, ts, consumer):
        self.my_log('start pull method with utc timestamp %s' % (dt.utcfromtimestamp(ts / 1000.0).strftime('%Y-%m-%d %H:%M:%S')))

        for pid, cmd_line in get_pids():
            config = self.get_config(cmd_line)
            if not config:
                self.my_log('unknown config for zookeeper process, pid: %s, cmd_line: %s' % (pid, cmd))
                continue

            instance = self.config2instance[config]
            try:
                zk_status = self.get_zk_status(pid, config)
            except Exception as e:
                self.my_log("can't get zookeeper stats for instance %s, %s %s" % (instance, type(e), e))
                continue

            consumed_sensors = 0
            for sensor in zk_status:
                try:
                    # могут попасться не числовые значения, просто пропускаем
                    consumer.gauge({'sensor': sensor, 'instance': instance}, ts, float(zk_status[sensor]))
                    consumed_sensors += 1
                except:
                    pass

            self.my_log('parsed zookeeper status for %s, %d sensors consumed' % (instance, consumed_sensors))

        self.my_log('end pull method')


def get_pids():
    """
    возвращает список из пар (pid, командная строка запуска)
    """
    try:
        ps_list = check_output(['/bin/ps', '-e', '-o', 'pid,cmd'], stderr=STDOUT)
        procs = [x.strip() for x in ps_list.split('\n')]
        return [(x[:x.find(' ')], x) for x in procs if ' /usr/bin/java' in x and '/etc/zookeeper' in x]
    except:
        return []


def applicable():
    return bool(get_pids())


# эта часть выполняется только при генерации конфигов (запускается системным python)
# и никак не связана со сбором метрик (запускается встроенным solomon-agent python)
if __name__ == '__main__':
    import sys
    import json

    sys.path.extend(['share/dt_solomon', '/usr/local/share/dt_solomon'])
    from dts_common import service_config, python2_pull_config, system_pull_config

    # пути к файлу и директории - параметры подставит генератор конфига
    file_path = sys.argv[1]
    module_pkg = sys.argv[2]
    # имя сервиса генерируется из имени файла, для my_module.py будет my-module
    service = sys.argv[3]

    if not applicable():
        print '{}'
        sys.exit(0)

    config = [
        # "контейнер" для однотипных сенсоров https://wiki.yandex-team.ru/solomon/userguide/datamodel/
        # solomon будет забирать сенсоры по hostname:.../...?project=...&service=...
        service_config(
            service=service,
            project='direct', 
            pull_interval='15s',
            modules=[
                python2_pull_config(file_path, module_pkg, "ZookeeperPullModule", params={}) # тут можно передать специфичные параметры, например, какие метрики нужны на данном хосте
            ],
            labels=[] # общие метки для всех модулей, почти никогда не нужны
        ),
    ]

    print json.dumps(config)
