#!/usr/bin/perl
#
use warnings;
use strict;
use utf8;

use lib::abs;

use my_inc ('../..', for => 'api/t');

=pod

    $Id:$

=head1 NAME

    wsdl_print.pl

=head1 SYNOPSIS

    wsdl_print.pl [ --tab=4 ] [ --version=5 ] service [ element, .. ]
    wsdl_print.pl --services

=head1 DESCRIPTION

    Выдает псевдо-json по заданному сервису и элементу внутри, в случае
    если элемент не указан выводятся структуры запроса и ответа для всех методов сервиса

        * tab - размер отступа в пробелах, по умолчанию 4
        * version - версия API, по умолчанию 5
        * services - выводит список сервисов и их операций
        * params - верхний элемент выводится как params, в случае если указан
            только сервис то список операций выводится в нотации
            { "method" : "operation", "params" : {..} }
            { "result" : {} }

    Пример
        ./wsdl_print.pl adgroups GetRequest GetResponse

=cut

use Getopt::Long;

use API::Services;
use API::WSDLFiles qw/wsdl_files xsd_definitions/;
use WSDL::JSON::Schema;
use WSDL::JSON::Schema::Printer qw/schema_dump/;

binmode(STDOUT, ":utf8");

my $version = 5;
my ($tab, $list_services, $as_params, $tab_s);
GetOptions(
    "params" => \$as_params,
    "services" => \$list_services,
    "version=i" => \$version,
    "tab=i" => \$tab
);

$WSDL::JSON::Schema::Printer::TAB = join('', map { ' ' } 1..$tab) if $tab;

$tab_s = $WSDL::JSON::Schema::Printer::TAB;

my ($service, @elements) = @ARGV;
usage() unless $list_services || $service;

my $services = API::Services->new( [ wsdl_files($version) ], [ xsd_definitions($version) ]);
my $nss = $services->namespaces;

if($list_services) {
    my $service_operations;
    $services->map_thru_service_operation_ns_parts(sub {
        my ($service_name, $operation, $ns, $request_element, $response_element) = @_;
        $service_operations->{$service_name} //= { ns => $ns, operations => [] };
        push @{$service_operations->{$service_name}{operations}}, { name => $operation, request => $request_element, response => $response_element };
    });
    foreach my $s (keys %$service_operations) {
        print qq~$s \{$service_operations->{$s}{ns}\}\n~;
        foreach my $o (@{$service_operations->{$s}->{operations}}) {
            print qq~\t$o->{name}\t$o->{request}\t$o->{response}\n~;
        }
        print "\n";
    }
    exit;
}

if(@elements) {
    for my $element (@elements) {
        builder_element_print(
            WSDL::JSON::Schema->new( $nss, "http://api.direct.yandex.com/v$version/" . lc($service) ), $element
        );
    }
} else {
    $services->map_thru_service_operation_ns_parts(sub {
        my ($service_name, $operation, $ns, $request_element, $response_element) = @_;
        return unless lc($service_name) eq lc($service);
        my $builder = WSDL::JSON::Schema->new( $nss, $ns );
        print "===" . ucfirst($service) . '.' . ucfirst($operation) . "===\n";

        print "{\n${tab_s}\"method\" : \"$operation\",\n" if $as_params;
        builder_element_print($builder, $request_element);
        print (($as_params ? '}' : '' ) . "\n");

        print "{\n" if $as_params;
        builder_element_print($builder, $response_element, 1);
        print (($as_params ? '}' : '' ) . "\n");

        print "\n";
    });
}

sub builder_element_print {
    my ($builder, $element, $is_reponse) = @_;
    my $schema = $builder->build($element);
    my $title = $as_params ? ($is_reponse ? 'result' : 'params') : $element;
    print schema_dump( $title => $schema);
}

sub usage {
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8");
    exit(1);
}

1;
