#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use my_inc ('../..', for => 'api/t');

=pod

    $Id$

=head1 NAME

    xml2perl.pl

=head1 SYNOPSIS

    Скрипт для конвертации запросов в API5 из отчетов в aqua в curl-скрипт
    --file  - получить запрос из файла
    --help  - эта справка
    --ts    - запустить на ТС
    --ts2   - запустить на ТС2
    --prod  - запустить на продакшне
    -r      - или --run не выводить скрипт, а сразу запустить

    По умолчанию из запроса полученного на STDIN выводится сприпт для запуска
    запроса на текущей бете, пример запроса оможно посмотреть в POD под
    заголовком example

    Примеры:
    xml2perl.pl -f req_file > curl.sh # чтобы потом поправить и запустить
    xml2perl.pl -f req_file -r # сразу запустить
    xml2perl.pl --ts < req_file > ts_curl.pl # скрипт для запуска на тс

=cut

package r;

use XML::LibXML;
use XML::LibXML::Simple qw/XMLin/;

use API::Request::SOAPAction;

use Mouse;

has $_ => (is => 'ro', isa => "Str") foreach (qw/service headers params_container/);
has $_ => (is => 'ro', isa => "Maybe[Str]") foreach (qw/language auth client/);

has params => (is =>'ro', isa => 'Ref');

sub BUILDARGS {
    my $class = shift;
    my $request = shift;

    $request =~ /REQUEST: (<\?xml.+>\s*$)/s or die "no xml found in request";
    my $xml_string = $1;
    $request =~ /^URL: (.+)\s*$/m or die 'no url';
    my $service = API::Request::SOAPAction->new("$1/get")->service;

    $request =~ /^HEADERS:\s+\{(.+)\}\s*$/m or die 'no headers';
    my $headers = $1;
    my $language = $class->_from_header($headers, 'Accept-Language');
    my $auth = $class->_from_header($headers, 'Authorization');
    my $client = $class->_from_header($headers, 'Client-Login');

    my $dom = XML::LibXML->load_xml(
        string => $xml_string,
        recover => 2
    );

    my $body = $dom->getElementsByLocalName('Body')->[0];
    my ($params_container) = (grep { $_->localName} $body->childNodes);
    my $params = XMLin($params_container->toString);

    return {
        params_container => $params_container->localName,
        params => $params,
        service => $service,
        language => $language,
        auth => $auth,
        client => $client,
    };

}

sub _from_header {
    my $self = shift;
    my $header = shift;
    my $name = shift;

    if($header =~ /$name=\[(.+?)\]/) {
        return $1;
    }
    return;
}

__PACKAGE__->meta->make_immutable;

1;

package main;

use Cwd;
use File::Slurp qw/read_file/;
use Getopt::Long;
use Pod::Usage;
use Template;
use XML::Twig;

use API::WSDLFiles qw/wsdl_files xsd_definitions/;
use API::Services;
use API::Version;

my ($ts, $ts2, $prod, $run, $request_file);
GetOptions(
    'ts+' => \$ts,
    'ts2+' => \$ts2,
    'prod+' => \$prod,
    'run+' => \$run,
    'file=s' => \$request_file
    ) || pod2usage();
pod2usage() if @ARGV;

cwd =~ m~\/beta\..+?\.(\d+)\/~ or die 'not a beta';

my $url;
if($ts) {
    $url = 'https://api.test.direct.yandex.ru:14443/api/v5'
} elsif ($ts2) {
    $url = 'https://test2-direct.yandex.ru:14443/api/v5'
} elsif ($prod) {
    $url = 'https://api.direct.yandex.ru/api/v5'
} else { # beta
    my $beta=$1;
    `hostname` =~ /ppcdev(\d+)/ or die 'wrong service';
    my $hostname = "beta$1";

    $url = "https://$beta.$hostname.direct.yandex.ru:14443/api/v5";
}

my $request_text = get_request();

my $r = r->new($request_text);

my $services = API::Services->new(
    [ wsdl_files($API::Version::number) ],
    [ xsd_definitions($API::Version::number) ]
);

my $operation = $services->operation_by_input_body_name($r->service, $r->params_container);

my $pretifier = XML::Twig->new(pretty_print => 'indented');
$pretifier->parse(
    $services->request_to_xml($r->service, $operation, $r->params)
);

my $request_xml = $pretifier->sprint;

my $curl_script;
my $template = join('', <DATA>);
my $tt = Template->new();

$tt->process(
    \$template,
    {
        service => $r->service,
        operation => $operation,
        language => $r->language,
        auth => $r->auth,
        client => $r->client,
        xml => $request_xml,
        url => $url
    },
    \$curl_script
) or die $tt->error;

if($run) {
    system($curl_script);
} else {
    print "\n\n\n$curl_script\n";
}

sub get_request {
    if($request_file) {
        return read_file($request_file);
    }
    return join('', <STDIN>);
}

1;


=head2 example
    URL: https://8791.beta1.direct.yandex.ru:14443/api/v5/adgroups
    HEADERS: {Accept-Language=[en], Authorization=[Bearer 821d25c7ecb54bb8a85e0b9433c92d6f]}
    REQUEST: <?xml version="1.0" encoding="UTF-8"?><S:Envelope xmlns:S="http://schemas.xmlsoap.org/soap/envelope/">
      <SOAP-ENV:Header/>
      <S:Body>
        <ns2:GetRequest>
          <Page>
            <Limit>0</Limit>
          </Page>
          <SelectionCriteria>
            <CampaignIds>351883216</CampaignIds>
          </SelectionCriteria>
          <FieldNames>Id</FieldNames>
        </ns2:GetRequest>
      </S:Body>
    </S:Envelope>
=cut

__DATA__
curl \
     -k \
    -H "SOAPAction: 'http://direct.yandex.com/api/v5/[% service %]/[% operation %]'" \
    -H 'Authorization: [% auth %]' \
    [%- IF client %]
    -H 'Client-Login: [% client %]' \
    [%- END -%]
    [%- IF language %]
    -H 'Accept-Language: [% language %]' \
    [%- END %]
    -d '[% xml %]' \
    "[% url %]/[% service %]"


