package API::Authorization;

use Direct::Modern;

=pod

    $Id$

=head1 NAME

    API::Authorization

=head1 SYNOPSIS


    my $auth = API::Authorization->new(
        token => $request->token,
        remote_address => $request->remote_ip,
        request_id => $request->id
    );
    my $bool = $auth->is_token_authenticated; # проверка токета
    my $bool = $auth->rbac_authenticated; # проверка прав в rbac
    my $role = $auth->role; # роль оператора

    $auth = $auth->fake_login_by_request($fake); # объект авторизации для fake-клиента из запроса

    my $user = $auth->operator_user; # Authorization::User для оператора
    my $login = $auth->chief_rep_login; # Логин главного представителя
    my $chief = $auth->chief_rep_user; # Authorization::User для главного представителя

    # Authorization::User сабклиента с вычислением главного представителя и проверкой
    # прав на него от оператора в RBAC
    my $subclient = $auth->subclient_by_login($request->login);

    my $operator_uid = $auth->operator_uid; # UID авторизованного по токену пользователя

=head1 DESCRIPTION

    Класс для реалзиации авторизации проверки прав
    Термины
        * operator - оператора, тот пользователь что получен по токену;
        * chief_rep - главный представитель пользователя из токена;
        * subclient - клиент над данными которого производится операция оператором.

    Подклассы
    Authorization::CheckToken - проверка и получение данных по токену в пасспорте
    Authorization::RBAC - все что связано с RBAC в авторизации
    Authorization::User - объектная обертка на данными пользвателя, умеет сама
                            получать данные пользователя по uid и обновлять карму
    Authorization::User::Short - тоже, но только с полями uid login ClientID и
                                без дополнительных опций
    Authorization::MouseIpType - Mouse тип для IP-адресов


=head1 METHODS

=cut

use Yandex::I18n;

use Settings; # qw/$Settings::KARMA_API_AUTOBLOCK/;

use Client qw/get_client_brand_id/;
use Direct::Errors::Messages;
use IpTools qw/is_ip_in_networks/;
use Primitives qw/get_user_info/;
use TextTools qw/normalize_login/;

use API::Authorization::MouseIpType;

use API::Authorization::RBAC;
use API::Authorization::CheckToken;
use API::Authorization::TokenFormat;
use API::Authorization::User;

use Mouse;
use Mouse::Util::TypeConstraints;

subtype 'token_bearer' => as 'Str' => where { API::Authorization::TokenFormat::is_valid_token($_) };
enum relation_type => ('freelancer', 'mcc', '');

has token => (
    is => 'ro',
    isa => 'token_bearer',
    required => 1,
);

has is_token_persistent => (
    is => 'ro',
    isa => 'Bool',
    default => sub { 0 }
);

has request_id => (
    is => 'ro',
    isa => 'Int',
    required => 1,
);

has remote_address => (
    is => 'ro',
    isa => 'ip',
    required => 1,
);


has application_id => (
    is => 'rw',
    isa => 'token_bearer',
);

# вычисляется внутри _token_auth_info
# не можем вычислять уже после из результата _token_auth_info
# т.е. объект оператора нужен нам в этом методе для получения объекта rbac
has operator_uid => (
    is => 'rw',
    isa => 'Int',
    predicate => 'has_operator_uid',
);

has operator_user => (
    is => 'ro',
    isa => 'API::Authorization::User',
    lazy => 1,
    default => sub {
        my $self = shift;
        die 'cant run before operator_uid set'
                unless $self->has_operator_uid;
        return API::Authorization::User->new($self->operator_uid);
    }
);

has chief_rep_user => (
    is => 'ro',
    isa => 'API::Authorization::User',
    lazy => 1,
    default => sub {
        return API::Authorization::User->new(shift->chief_rep_uid);
    }
);

has operator_client_relation_type => (
    is => 'ro',
    isa => 'relation_type',
    lazy => 1,
    default => sub { '' },
    writer => '_set_operator_client_relation_type',
);

# в случае если авторизованный пользователь не равен главному представителю
# то вычисляется главный представиль и для всех последующих проверок
# прав используется он. Реквизиты авторизованного же пользователя сохраняются
# как operator_uid и operator_login на случай если они понадобятся

has _token_auth_info => (
    is => 'ro',
    isa => 'Maybe[HashRef]',
    lazy => 1,
    predicate => 'token_authorization_done',
    default => sub {
        my $self = shift;

        my $auth = $self->is_token_persistent
                    ? API::Authorization::CheckToken->authorize_by_persistent_token($self->token, $self->remote_address)
                    : API::Authorization::CheckToken->authorize($self->token, $self->request_id);

        if($auth && $auth->{uid}) {
            $self->operator_uid($auth->{uid});
            my $user = $self->operator_user;

            $self->application_id($auth->{application_id});

            # токен может быть на пользователя, которого нет в директе
            if($user->found) {
                $self->_auth_hash_fix_operator($auth);
            }
            return $auth;
        }
        return undef;
    }
);

has _rbac => (
    is => 'ro',
    isa => 'API::Authorization::RBAC',
    lazy => 1,
    default => sub {
        my $self = shift;
        # иначе бегаем по кругу
        die "no _token_auth_info" unless $self->token_authorization_done;

        return API::Authorization::RBAC->new(
            user => $self->operator_user,
            remote_address => $self->remote_address,
            request_id => $self->request_id,
        );
    }
);

=head2 check_api_access

Возвращает ничего, если доступ есть, и Direct::Defect, который надо отдать,
если доступа нет.

=cut

sub check_api_access {
    my $self = shift;
    my $operator = $self->operator_user;

    # Проверяем годность пользователя для работы с API в RBAC
    unless ( $self->rbac_authenticated ) {
        if ( $self->rbac_check_status == 3 ) {
            return error_NoRights(iget("Доступ возможен только из внутренней сети Яндекса"));
        }

        return error_NoRights();
    }

    if ($operator->statusBlocked eq 'Yes') {
        return error_AccessToApiDenied_AccountBlocked();
    }

    if ( !$self->_rbac->has_api_access ) {
        return error_AccessToApiDenied_AccountBlocked();
    }

    my $api_allowed_ips = $operator->api_allowed_ips;
    if ( defined $api_allowed_ips && $api_allowed_ips ne '' ) {
        if ( ! is_ip_in_networks( $self->remote_address, $api_allowed_ips ) ) {
            return error_AccessToApiDenied(iget("Пользователь запретил доступ с данного IP-адреса"));
        }
    }

    return error_AccessToApiDenied('Доступ для пользователей услуги Настройка Яндекс.Директа запрещен') if $self->is_role_client && $operator->is_ya_agency_client;

    return;
}


sub _auth_hash_fix_operator {
    my ($self, $auth) = @_;
    if( my $uid = $auth->{operator_uid} = $auth->{uid} ) {
        $auth->{operator_login} = $auth->{login};

        my $rbac = API::Authorization::RBAC->new(
            user => $self->operator_user,
            remote_address => $self->remote_address,
            request_id => $self->request_id,
        );

        my $chief_rep = get_user_info(
            $rbac->uid_to_chief_rep_uid($uid)
        );

        if($chief_rep->{uid} != $uid) {
            $auth->{login} = $chief_rep->{login};
            $auth->{uid} = $chief_rep->{uid};
        }
    }
    return $auth;
}

sub chief_login_by_rep_login {
    my ($self, $login) = @_;
    my $uid = $self->uid_by_login($login) or return;
    return (get_user_info(
        $self->_rbac->uid_to_chief_rep_uid($uid)
    ) // {} )->{login};
}

sub fake_login_by_request {
    my ($self, $request) = @_;

    my $login = $request->fake_login;
    my $uid = $self->uid_by_login_direct($login) or return;
    my $auth = $self->_auth_hash_fix_operator({
        uid => $uid,
        login => $login,
        application_id => '00000000000000000000000000000000',
    });

    return $self->fake_new($uid, $login, {
        request_id => $request->id,
        remote_address => $request->remote_ip,
        _token_auth_info => $auth
    });
}

sub fake_new {
    my ($self, $uid, $login, $params) = @_;
    my $fake_self = $self->new(
        request_id => '1',
        remote_address => '127.0.0.1',
        token => '00000000000000000000000000000000',
        application_id => '00000000000000000000000000000000',
        operator_uid => $uid,
        _token_auth_info => {
            uid => $uid,
            login => $login,
            operator_uid => $uid,
            operator_login => $login,
        },
        %$params
    );

    return $fake_self;
}

sub role_by_uid {
    my ($self, $uid) = @_;
    $self->_rbac->role_by_uid($uid);
}

sub operator_login { shift->_token_auth_info->{operator_login} }

sub chief_rep_login { shift->_token_auth_info->{login} }
sub chief_rep_uid { shift->_token_auth_info->{uid} }
sub application_id { shift->_token_auth_info->{application_id} }

sub is_token_authorized {
    my $self = shift;
    return
        ( $self->_token_auth_info and defined $self->_token_auth_info->{uid} )
        ? 1
        : 0;
}

sub rbac_authenticated {
    shift->_rbac->authenticated;
}

sub rbac_check_status {
    return shift->_rbac->login_check_error_code;
}

sub rights {
    return shift->_rbac->rights;
}

sub role {
    return shift->rights->role;
}

=head2 is_role_client

    true если авторизованный пользотель - прямой клиент (не сотрудник яндекса и не агентство)

=cut

sub is_role_client { return shift->rights->role eq 'client' }

=head2 is_role_agency

    true если авторизованный пользотель - представитель агентства

=cut

sub is_role_agency { return shift->rights->role eq 'agency' }

=head2 is_role_super

    true если авторизованный пользотель является суперпользователем

=cut

sub is_role_super { return shift->rights->role eq 'super' }

=head2 is_role_superreader

    true если авторизованный пользотель является суперпользователем

=cut

sub is_role_superreader { return shift->rights->role eq 'superreader' }

=head2 subclient_by_login($login)

    Проверяет что у оператора пользователя есть права на данный
    логин (его главного представителя). Наример клиент сервисируемый
    авторизованным агенством или авторизован супер.
    В случае успеха проверки возвращает главного представителя сабклиента

=cut

sub subclient_by_login {
    my $self = shift;
    my $login = shift or die 'no login specified';
    my $uid = $self->uid_by_login_direct($login) or return;
    my $chief_rep_uid = $self->_rbac->uid_to_chief_rep_uid($uid);
    return unless $self->_rbac->is_owner($chief_rep_uid);

    return API::Authorization::User->new($chief_rep_uid);
}

=head2 check_operator_and_client_relation($login)

    Позвращает признак того, что оператор является фрилансером,
    сопровождающим указанного клиента.

    При этом, если оператор - фрилансер и имеет доступ к запрошенному клиенту,
    в auth выставляем флаг operator_is_freelacer
=cut

sub check_operator_and_client_relation {
    my $self = shift;

    my $login = shift or die 'no login specified';
    my $client_client_id = $self->client_id_by_login($login) or return;
    my $relation_type = $self->_rbac->get_client_relation_type($client_client_id);
    if ($relation_type){
        $self->_set_operator_client_relation_type($relation_type);

        return 1;
    }

    return 0;
}

=head2 client_id_by_login($login)

    Получить ClientID по логину

=cut

sub client_id_by_login { return Primitives::get_clientid(login => $_[1]) }

=head2 uid_by_login_direct($login)

    Получить uid по логину в Директе, без хождения в пасспорт

=cut

sub uid_by_login_direct { return Primitives::get_uid(login => $_[1]) }


=head2 uid_by_login($login)

    Получить uid по логину, сходить в пасспорт если логина нет в директе
    Логин в директе может быть не найден если хранится в БД не нормализованным
    поэтому иногда нужно сходить за ним в пасспорт.

=cut

sub uid_by_login { return Primitives::get_uid_by_login2($_[1]) }

=head2 login_exists($login)

    Проверить что пользователь с таким логином действительно существует

=cut

sub login_exists {
    my ($self, $login) = @_;
    return $self->uid_by_login_direct($login) ? 1 : 0;
}

=head2 match_operator_or_chief_rep_login($login)

    Проверяет что логин (или логин главного представителя) совпадает с логином
    оператора или его главным представителем

    Параметры:
        $login - строка логин

    Ответ:
        true если совпадают, иначе false

=cut

sub match_operator_or_chief_rep_login {
    my ($self, $login) = @_;

    return unless $login;

    return (
        $self->are_logins_equal($login, $self->operator_login),
        $self->are_logins_equal($login, $self->chief_rep_login),
        $self->are_logins_equal($self->chief_login_by_rep_login($login), $self->chief_rep_login)
    );
}

=head2 are_logins_equal($login1, $login2)

    Сравнивает две строки, возвращает true если в обоих строках один и тот же логин

=cut

sub are_logins_equal {
    my ($self, $login1, $login2) = @_;

    return unless defined $login1 && defined $login2;

    return lc(normalize_login($login1)) eq lc(normalize_login($login2));
}

=head2 get_brand_chief_rep_user_by_clientid($client_id)

    Для $client_id ClientID клиента возвращает объект пользователя главного представителя
    главного клиента брэнда если $client_id под брендом, иначе undef

=cut

sub get_brand_chief_rep_user_by_clientid {
    my ($self, $client_id) = @_;

    my $brand_ClientID = get_client_brand_id( $client_id );
    if ($brand_ClientID) {
        return API::Authorization::User->new(
            $self->_rbac->client_id_to_chief_rep_uid($brand_ClientID)
        );
    }
    return;
}
__PACKAGE__->meta->make_immutable();

1;
