package API::Authorization::RBAC;

use Direct::Modern;

=pod

    $Id$

=head1 NAME

    API::Authorization::RBAC

=head1 SYNOPSIS

    my $rbac = API::Authorization::RBAC->new(
        user => $operator_user,
        remote_address => $request->remote_address,
        request_id => $request->id
    );

    my $bool = $rbac->authenticated; # Boolean истино если авторизация в рбак пройдена
    my $rights = $rbac->rights; # права авторизованного в рбак пользователя
    my $code = $rbac->login_check_error_code; # ошибка проверки прав в RBAC если есть

    my $bool = $rbac->is_owner($uid); # праверка имеет ли текущий оператор права на пользователя с $uid

    my $UID = $rbac->uid_to_chief_rep_uid($uid); # преобразует uid пользователя в UID его главного представителя

=head1 DESCRIPTION

    Интерфейс к RBAC2, вся работа из авторизации с RBAC2 ведется через данный класс

    Является интерфейсом к классам
    API::Authorization::RBAC::LoginCheck - проверка прав, обертка над rbac_login_check
    API::Authorization::RBAC::LoginCheck::Rights - объект с правами


=head1 METHODS

=cut

use Mouse;

use RBACDirect qw/rbac_is_owner rbac_can_use_api rbac_get_managers_of_client rbac_has_agency/;
use RBACElementary qw/
    rbac_get_chief_rep_of_client
    rbac_get_chief_rep_of_agency_rep
    rbac_get_chief_rep_of_client_rep
    rbac_who_is
/;

use API::Authorization::MouseIpType;
use API::Authorization::RBAC::LoginCheck;

use constant ROLE_AGENCY => 'agency';

has user => (
    is => 'ro',
    isa => 'API::Authorization::User',
    required => 1,
);

has uid => (
    is => 'ro',
    isa => 'Int',
    lazy => 1,
    default => sub {
        shift->user->uid
    }
);

has remote_address => (
    is => 'ro',
    isa => 'ip',
    required => 1,
);

has request_id => (
    is => 'ro',
    isa => 'Int',
    required => 1,
);

has _login_checker => (
    is => 'ro',
    isa => 'API::Authorization::RBAC::LoginCheck',
    lazy => 1,
    default => sub {
        my $self = shift;
        return API::Authorization::RBAC::LoginCheck->new(
            user => $self->user,
            remote_address => $self->remote_address,
        );
    }
);

=head2 current_role

    Роль пользователя uid которого указан при инициализации объекта

=cut

has current_role => (
    is => 'ro',
    lazy => 1,
    default => sub {
        my $self = shift;
        $self->role_by_uid($self->uid);
    });

=head2 is_under_agency

    True если клиент с ролью client и находится под агенством

=cut

has is_under_agency => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub {
    my $self = shift;
    return $self->current_role eq 'client' && rbac_has_agency(undef, $self->uid) ? 1 : 0;
});

=head2 is_under_manager

    True если обслуживается менеджером яндекса

=cut

has is_under_manager => ( is => 'ro', isa => 'Bool', lazy => 1, default => sub {
    my $self = shift;
    return scalar @{rbac_get_managers_of_client(undef, $self->uid)} ? 1 : 0;
});

=head2 has_api_access

    Проверяет включен ли доступ к API (api_enabled)

=cut

sub has_api_access {
    my $self = shift;
    return rbac_can_use_api(
        undef, {
            uid => $self->uid,
            UID => $self->uid,
            api_enabled => $self->user->api_enabled,
            ClientID => $self->user->ClientID,
            client_have_agency => $self->rights->client_have_agency
        }
    );
}

=head2 authenticated

    Bool, вернет true если авторизация прошла успешна

=cut

sub authenticated {
    shift->_login_checker->authenticated_ok;
}

=head2 rights

    Права пользователя в RBAC, см. RBAC2::DirectChecks::rbac_login_rights

=cut

sub rights {
    return shift->_login_checker->rbac_rights;
}

=head2 login_check_error_code

    Код ошибки авторизации

=cut

sub login_check_error_code {
    return shift->_login_checker->error_code;
}

=head2 is_owner($uid)

    Проверяет что текущий пользователь владеет указанным $uid, например $uid
    принадлежит авторизованному агенству

=cut

sub is_owner {
    my $self = shift;
    my $uid = shift;
    return rbac_is_owner(undef, $self->uid, $uid);
}

=head2 get_client_relation_type($clientid)

    Возвращает тип связи между оператором и заданным клиентом, если она есть

=cut

sub get_client_relation_type {
    my ($self, $client_id) = @_;
    return RBACDirect::rbac_get_client_relation_type(undef, $self->user->ClientID, $client_id);
}

=head2 uid_to_chief_rep_uid($uid)

    Возвращает UID главного преставителя для указанного $uid

=cut

sub uid_to_chief_rep_uid {
    my $self = shift;
    my $uid = shift;
    return $self->role_by_uid($uid) eq ROLE_AGENCY
        ? rbac_get_chief_rep_of_agency_rep($uid)
        : rbac_get_chief_rep_of_client_rep($uid);
}

=head2 client_id_to_chief_rep_uid($ClientID)

    Возвращает UID главного преставителя для указанного $ClientID

=cut

sub client_id_to_chief_rep_uid {
    my $self = shift;
    my $ClientID = shift;
    return rbac_get_chief_rep_of_client(
        $ClientID
    );
}

=head2 role_by_uid($uid)

    Роль по $uid (client, super, place, manager, superreader, teamleader и т.д.)

=cut

sub role_by_uid {
    my ($self, $uid) = @_;
    return rbac_who_is(undef, $uid);
}

__PACKAGE__->meta->make_immutable();

1;

__END__


